package com.liveperson.infra.preferences

import android.content.Context
import android.content.SharedPreferences
import com.liveperson.infra.CampaignInfo
import com.liveperson.infra.controller.DBEncryptionHelper
import com.liveperson.infra.errors.ErrorCode.ERR_00000151
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.model.PushMessage
import com.liveperson.infra.utils.EncryptionVersion
import org.jetbrains.annotations.TestOnly
import java.util.*

/**
 * Handles persistent storage of PushMessage object created when received push notification.
 * At the moment we store push messages we receive from proactive campaign messaging as a part of outbound
 * messaging feature.
 */
object PushMessagePreferences {

    private const val TAG = "PushMessagePreferences"
    private const val KEY_ID = "id"
    private const val KEY_ALL_IDS = "all_ids"
    private const val KEY_TIME_CREATED = "time_created"
    private const val KEY_MESSAGE = "message"
    private const val KEY_TITLE = "title"
    private const val KEY_AGENT_NAME = "agentName"
    private const val KEY_BACKEND_SERVICE = "backendService"
    private const val KEY_COLLAPSE_KEY = "collapse_key"
    private const val KEY_CONVERSATION_ID = "conversationId"
    private const val KEY_LE_ENGAGEMENT_ID = "leEngagementId"
    private const val KEY_LE_CAMPAIGN_ID = "leCampaignId"
    private const val KEY_IS_OUTBOUND_CAMPAIGN = "isOutboundCampaign"
    private const val KEY_LOOK_BACK_PERIOD = "lookBackPeriod"
    private const val KEY_AGENT_PID = "agentPid"
    private const val KEY_TRANSACTION_ID = "transactionId"
    private const val KEY_EVENT = "event"

    private const val LP_PUSH_MESSAGE_SHARED_PREF_FILENAME = "lp_push_message_shared_pref"

    private const val KEY_IS_PUSH_NOTIFICATION_CLICKED = "is_push_notification_clicked"
    private const val KEY_CLICKED_NOTIFICATION_ID = "clicked_notification_id"
    private const val KEY_PUSH_TYPE = "key_push_type"

    private var preferences: SharedPreferences? = null

    /**
     * Initialize PushMessage shared preferences.
     */
    fun init(applicationContext: Context) {
        if (preferences == null) {
            preferences = applicationContext.getSharedPreferences(
                    LP_PUSH_MESSAGE_SHARED_PREF_FILENAME, Context.MODE_PRIVATE)
        }
    }

    /**
     * Stores a PushMessage object for a Brand in Preferences (encrypted)
     */
    fun setCachedPushMessage(brandId: String, pushMessage: PushMessage) {

        if (preferences == null) {
            LPLog.e(TAG, ERR_00000151, "setCachedPushMessage: PushMessage preferences is not initialized")
            return
        }
        val editor = preferences?.edit()
        val messageId = pushMessage.pushMessageId

        editor?.putString(generateKey(brandId, messageId, KEY_ID),
                DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, messageId))
        editor?.putLong(generateKey(brandId, messageId, KEY_TIME_CREATED), pushMessage.timeCreated)
        editor?.putString(generateKey(brandId, messageId, KEY_COLLAPSE_KEY),
                DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, pushMessage.collapseKey))
        editor?.putString(generateKey(brandId, messageId, KEY_CONVERSATION_ID),
                DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, pushMessage.conversationId))
        editor?.putString(generateKey(brandId, messageId, KEY_MESSAGE),
                DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, pushMessage.message))
        editor?.putString(generateKey(brandId, messageId, KEY_BACKEND_SERVICE),
                DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, pushMessage.backendService))
        editor?.putString(generateKey(brandId, messageId, KEY_AGENT_NAME),
                DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, pushMessage.from))
        editor?.putString(generateKey(brandId, messageId, KEY_AGENT_PID),
                DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, pushMessage.agentPid))
        editor?.putString(generateKey(brandId, messageId, KEY_TRANSACTION_ID),
                DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, pushMessage.transactionId))
        editor?.putString(generateKey(brandId, messageId, KEY_TITLE),
                DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, pushMessage.title))
        editor?.putString(generateKey(brandId, messageId, KEY_EVENT),
                DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, pushMessage.messageEvent))


        if (pushMessage.campaignInfo != null) {
            editor?.putString(generateKey(brandId, messageId, KEY_LE_ENGAGEMENT_ID),
                    DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, pushMessage.campaignInfo.engagementId.toString()))
            editor?.putString(generateKey(brandId, messageId, KEY_LE_CAMPAIGN_ID),
                    DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, pushMessage.campaignInfo.campaignId.toString()))
            editor?.putBoolean(generateKey(brandId, messageId, KEY_IS_OUTBOUND_CAMPAIGN), pushMessage.campaignInfo.isOutboundCampaign)
            editor?.putLong(generateKey(brandId, messageId, KEY_LOOK_BACK_PERIOD), pushMessage.lookBackPeriod)
        }
        editor?.apply()
        addIdToSet(messageId, brandId)
    }

    /**
     * Retrieves the PushMessage object for a Brand (decrypted)
     */
    fun getCachedPushMessage(messageId: String, brandId: String): PushMessage? {
        if (preferences == null) {
            LPLog.e(TAG, ERR_00000151, "getCachedPushMessage: PushMessage preferences is not initialized")
            return null
        }
        var pushMessage: PushMessage? = null

        val message = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                preferences?.getString(generateKey(brandId, messageId, KEY_MESSAGE), ""))

        if (!message.isNullOrEmpty()) {
            val agentName = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                    preferences?.getString(generateKey(brandId, messageId, KEY_AGENT_NAME), ""))
            pushMessage = PushMessage(brandId, agentName, message)

            pushMessage.backendService = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                    preferences?.getString(generateKey(brandId, messageId, KEY_BACKEND_SERVICE), ""))
            pushMessage.pushMessageId = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                    preferences?.getString(generateKey(brandId, messageId, KEY_ID), ""))
            pushMessage.timeCreated = preferences?.getLong(generateKey(brandId, messageId, KEY_TIME_CREATED), 0)
            pushMessage.collapseKey = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                    preferences?.getString(generateKey(brandId, messageId, KEY_COLLAPSE_KEY), ""))
            pushMessage.conversationId = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                    preferences?.getString(generateKey(brandId, messageId, KEY_CONVERSATION_ID), ""))

            pushMessage.lookBackPeriod = preferences?.getLong(generateKey(brandId, messageId, KEY_LOOK_BACK_PERIOD), -1)
            pushMessage.agentPid = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                    preferences?.getString(generateKey(brandId, messageId, KEY_AGENT_PID), ""))
            pushMessage.transactionId = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                    preferences?.getString(generateKey(brandId, messageId, KEY_TRANSACTION_ID), ""))
            val campaignId = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                    preferences?.getString(generateKey(brandId, messageId, KEY_LE_CAMPAIGN_ID), ""))
            val engagementId = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                    preferences?.getString(generateKey(brandId, messageId, KEY_LE_ENGAGEMENT_ID), ""))
            val isOutboundCampaign: Boolean = preferences?.getBoolean(generateKey(brandId, messageId, KEY_IS_OUTBOUND_CAMPAIGN), false)!!

            pushMessage.campaignInfo = CampaignInfo(campaignId.toLong(), engagementId.toLong(), isOutboundCampaign)
            pushMessage.title = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                    preferences?.getString(generateKey(brandId, messageId, KEY_TITLE), ""))
            pushMessage.messageEvent = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                    preferences?.getString(generateKey(brandId, messageId, KEY_EVENT), ""))
        }

        return pushMessage
    }

    /**
     * Retrieve stored Push Message using message Id
     */
    fun getCachedPushWelcomeMessage(messageId: String?, brandId: String): String {
        return DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                preferences?.getString(messageId?.let { generateKey(brandId, it, KEY_MESSAGE) }, ""))
    }

    /**
     * Set push notification id of recently tapped push message
     */
    fun setPushMessageClicked(pushMessageId: String?, isClicked: Boolean) {
        if (preferences == null) {
            LPLog.e(TAG, ERR_00000151, "setPushMessageClicked: PushMessage preferences is not initialized")
            return
        }
        val editor = preferences?.edit()
        editor?.putBoolean(KEY_IS_PUSH_NOTIFICATION_CLICKED, isClicked)
        if (!pushMessageId.isNullOrEmpty()) {
            editor?.putString(KEY_CLICKED_NOTIFICATION_ID, pushMessageId)
        }
        editor?.apply()
    }

    /**
     * Check If user has navigated to conversation screen via push notification
     */
    fun isPushNotificationClicked(): Boolean {
        return preferences?.getBoolean(KEY_IS_PUSH_NOTIFICATION_CLICKED, false) ?: false
    }

    /**
     * Get recently tapped push notification id
     */
    fun getClickedNotificationId(): String? {
        val id = preferences?.getString(KEY_CLICKED_NOTIFICATION_ID, "") ?: ""
        LPLog.d(TAG, "getClickedNotificationId: Clicked notification id: " + LPLog.mask(id))
        return id
    }

    /**
     * Method used to receive the latest not expired push for brand.
     *
     * @param brandId - id of brand which emitted push notifications
     * @return push message id of latest not expired push notification or null otherwise
     */
    fun getLatestNotificationIdForBrand(brandId: String): String? {
        val allIds = getAllIds()
        return allIds?.asSequence()
            ?.mapNotNull { id -> getCachedPushMessage(id, brandId) }
            ?.filterNot { it.isExpired }
            ?.maxBy { it.timeCreated }
            ?.pushMessageId
    }

    fun setPushPlatform(pushType: String) {
        preferences?.edit()?.putString(KEY_PUSH_TYPE, pushType)?.apply()
    }

    fun getPushPlatform(): String {
        return preferences?.getString(KEY_PUSH_TYPE, "") ?: ""
    }

    /**
     * Return backend service name associated with push message
     */
    fun getBackendService(messageId: String?, brandId: String): String? {
        return DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                preferences?.getString(messageId?.let { generateKey(brandId, it, KEY_BACKEND_SERVICE) }, null))
    }

    /**
     * Return transaction id associated with push message
     */
    fun getTransactionId(messageId: String?, brandId: String): String? {
        return DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1,
                preferences?.getString(messageId?.let { generateKey(brandId, it, KEY_TRANSACTION_ID) }, null))
    }

    /**
     * Add new message id into list of ids
     */
    private fun addIdToSet(id: String, brandId: String) {
        val ids = preferences?.getStringSet(KEY_ALL_IDS, null)
        LPLog.d(TAG, "addIdToSet: Adding notification id: " + LPLog.mask(id) + " into the list.")
        if (ids.isNullOrEmpty()) {
            val newSet = HashSet<String>()
            newSet.add(id)
            preferences?.edit()?.putStringSet(KEY_ALL_IDS, newSet)?.apply()
        } else {
            ids.add(id)
            preferences?.edit()?.putStringSet(KEY_ALL_IDS, ids)?.apply()
        }
    }

    /**
     * Return all stored message ids
     */
    private fun getAllIds(): MutableSet<String>? {
        return preferences?.getStringSet(KEY_ALL_IDS, null)
    }

    /**
     * Generate a unique key for each push message using messageId
     */
    private fun generateKey(brandId: String, msgId: String, key: String): String {
        return "$key::$msgId::$brandId"
    }

    /**
     * Clear PushMessage data with respective messageId from preferences
     */
    fun clearPushMessage(brandId: String, messageId: String) {
        LPLog.d(TAG, "clearPushMessage: Clearing PushMessage with id: $messageId from preferences")
        preferences?.edit()?.remove(generateKey(brandId, messageId, KEY_ID))
                ?.remove(generateKey(brandId, messageId, KEY_TIME_CREATED))
                ?.remove(generateKey(brandId, messageId, KEY_COLLAPSE_KEY))
                ?.remove(generateKey(brandId, messageId, KEY_CONVERSATION_ID))
                ?.remove(generateKey(brandId, messageId, KEY_MESSAGE))
                ?.remove(generateKey(brandId, messageId, KEY_TITLE))
                ?.remove(generateKey(brandId, messageId, KEY_AGENT_NAME))
                ?.remove(generateKey(brandId, messageId, KEY_AGENT_PID))
                ?.remove(generateKey(brandId, messageId, KEY_TRANSACTION_ID))
                ?.remove(generateKey(brandId, messageId, KEY_EVENT))
                ?.remove(generateKey(brandId, messageId, KEY_BACKEND_SERVICE))
                ?.remove(generateKey(brandId, messageId, KEY_LE_ENGAGEMENT_ID))
                ?.remove(generateKey(brandId, messageId, KEY_LE_CAMPAIGN_ID))
                ?.remove(generateKey(brandId, messageId, KEY_IS_OUTBOUND_CAMPAIGN))
                ?.remove(generateKey(brandId, messageId, KEY_LOOK_BACK_PERIOD))?.apply()
    }

    /**
     * Iterate through all stored push message ids and delete respective data If they are expired.
     */
    private fun clearExpiredPushMessages(brandId: String) {
        val allIds = getAllIds()
        val editor = preferences?.edit()
        LPLog.d(TAG, "Stored Push Message Ids: " + LPLog.mask(allIds))
        if (allIds != null) {
            val iterator = allIds.iterator()

            while (iterator.hasNext()) {
                val id = iterator.next()
                val pushMessage = getCachedPushMessage(id, brandId)
                if (pushMessage?.isExpired == true) {
                    LPLog.d(TAG, "clearExpiredPushMessages: Found expired push message in prefs. id: " + LPLog.mask(id) + ". Removing related data.")
                    clearPushMessage(brandId, id)
                    iterator.remove() // remove expired id from all ids list
                }
            }
            editor?.putStringSet(KEY_ALL_IDS, allIds)?.apply() // Update all ids list
            editor?.apply()
        }
    }

    /**
     * Cleanup user notification actions and expired push messages.
     */
    fun cleanUp(brandId: String) {
        LPLog.d(TAG, "cleanUp: Cleaning up PushMessage preferences.")
        clearExpiredPushMessages(brandId)
        preferences?.edit()?.remove(KEY_IS_PUSH_NOTIFICATION_CLICKED)
                ?.remove(KEY_CLICKED_NOTIFICATION_ID)?.apply()
    }

    /**
     * Clear all entries from PushMessage preferences
     */
    fun clearAll() {
        LPLog.d(TAG, "clearAll: Clearing all data of PushMessage preferences")
        preferences?.edit()?.clear()?.apply()
    }

    /**
     * Set mocked shared preference test delegate.
     */
    @TestOnly
    fun setSharedPreferencesDelegate(preferencesDelegate: SharedPreferences?) {
        preferences = preferencesDelegate
    }
}