package com.liveperson.infra.utils;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000015;

import android.os.AsyncTask;
import android.os.Build;
import android.text.TextUtils;
import android.util.Patterns;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

import com.liveperson.infra.R;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.utils.patterns.PatternsCompat;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by maayank on 17/11/2016.
 * a util class for link preview feature (and for structure content)
 * receives a URL and parse the open graph (and other relevant) tags from it
 * it uses a callback to return the response
 * we use AsyncTask with threadPoolExecutor to optimize
 */
public class TextCrawler {

	private static final String TAG = "TextCrawler";
	private static final String KEY_HTML_TAG_HEAD_OPEN = "<head>";
	private static final String KEY_HTML_TAG_HEAD_CLOSE = "</head>";
	private static final String KEY_HTML_TAG_TITLE = "title";
	private static final String KEY_HTML_TAG_DESCRIPTION = "description";
	private static final String KEY_HTML_TAG_SITE_NAME = "site_name";
	private static final String KEY_EMPTY = "";
	private static final String KEY_HTML_TAG_IMAGE = "image";
	private static final String KEY_HTML_TAG_URL = "url";
	private static final int CONNECT_TIMEOUT_IN_MILLI = 1500;
	private static final int READ_TIMEOUT_IN_MILLI = 5000;
	private static final String HTTP_PROTOCOL_SHORT = "http";
	private static final String HTTP_PROTOCOL = "http://";
	private static final String HTTPS_PROTOCOL = "https://";
	private static final String WWW_SUBDOMAIN = "www.";

	private static final String TITLE_PATTERN = "<title(.*?)>(.*?)</title>";
	private static final String SCRIPT_PATTERN = "<script(.*?)>(.*?)</script>";
	private static final String METATAG_PATTERN = "<meta(.*?)>";
	private static final String METATAG_CONTENT_PATTERN = "content=\"(.*?)\"";

	private LinkPreviewCallback mCallback;
	private SourceContent mSourceContent;
	private GetCode mGetCode;
	private HttpURLConnection hc;

	// We want at least 2 threads and at most 4 threads (if that's the max we have) in the core pool,
	// preferring to have 1 less than the CPU count to avoid saturating
	// the CPU with background work
	private static final int CPU_COUNT = Runtime.getRuntime().availableProcessors();

	// the number of threads to keep in the pool, even if they are idle, unless {@code allowCoreThreadTimeOut} is set
	private static final int CORE_POOL_SIZE = Math.max(2, Math.min(CPU_COUNT - 1, 4));

	// the maximum number of threads to allow in the pool
	private static final int MAXIMUM_POOL_SIZE = CPU_COUNT * 2 + 1;

	// keepAliveTime when the number of threads is greater than the core, this is the maximum time that excess idle threads will wait for new tasks before terminating.
	private static final int KEEP_ALIVE_SECONDS = 2;

	//
	private static final TimeUnit KEEP_ALIVE_TIME_UNIT = TimeUnit.SECONDS;
	private final BlockingQueue<Runnable> mDecodeWorkQueue  = new LinkedBlockingQueue<>();


	/**
	 * makePreview - initialize the url to parse and the callback to use with the result
	 * Create a new TextCrawler object before invoke this method.
	 */
	public void makePreview(LinkPreviewCallback callback, String url) {
		LPLog.INSTANCE.d(TAG, "makePreview. callback = " + callback + " url = " + url);
		if (this.mCallback != null) {
			LPLog.INSTANCE.w(TAG, "makePreview(...) canceled. Make sure You recreated a new TextCrawler object before invoke this method");
			return;
		}
		this.mCallback = callback;
		mSourceContent = new SourceContent();

		if (mGetCode != null) {
			mGetCode.cancel(true);
			mGetCode = null;
		}
		mGetCode = new GetCode();
		// TODO Refactor this, prevent creating extra executors.
		ThreadPoolExecutor mDecodeThreadPool = new ThreadPoolExecutor(
				CORE_POOL_SIZE,       // Initial pool size
				MAXIMUM_POOL_SIZE,    // Max pool size
				KEEP_ALIVE_SECONDS,
				KEEP_ALIVE_TIME_UNIT,
				mDecodeWorkQueue
		);
		mGetCode.executeOnExecutor(mDecodeThreadPool, url);
	}

	/**
	 * Get html code - inner class that extends AsyncTask
	 * to parse the relevant tags and return a result on the main thread
	 * it will return an error in case the parsing isn`t possible
	 */
	public class GetCode extends AsyncTask<String, Void, Void> {

		/**
		 * Finds urls inside the text and return the matched ones
		 */
		GetCode() { }

		@Override
		protected void onPreExecute() {
			LPLog.INSTANCE.d(TAG,"onPreExecute "+ this);
			if (mCallback != null) {
				mCallback.onPre();
			}
			super.onPreExecute();
		}

		@Override
		protected void onPostExecute(Void result) {
			LPLog.INSTANCE.d(TAG, "onPostExecute " + this + " result " + result + " mSourceContent " + mSourceContent);
			if (mCallback != null) {
				mCallback.onPos(mSourceContent, isNull());
			}
			super.onPostExecute(result);
		}

		@Override
		protected Void doInBackground(String... params) {
			LPLog.INSTANCE.d(TAG, "doInBackground " + this + " params = " + Arrays.toString(params));
			parseHtml(params);
			return null;
		}

		/**
		 * Verifies if the content could not be retrieved
		 */
		public boolean isNull() {
			return !mSourceContent.isSuccess() && extendedTrim(mSourceContent.getHtmlCode()).equals(KEY_EMPTY);
		}
	}

	public static String prepareLink(String text) {
		if (text == null) {
			LPLog.INSTANCE.w(TAG, "matches: given text is null");
			return null;
		}
		String url = null;
		text = text.toLowerCase();
		text = text.replaceAll("\u200B", "");
		text = text.replaceAll("\u200C", "");
		text = text.replaceAll("\u200C", "");
		text = text.replaceAll("\uFEFF", "");

		Matcher matcher = PatternsCompat.AUTOLINK_WEB_URL.matcher(text);
		while (matcher.find()) {
			String string = matcher.group();
			// important - this code is to form a valid url to open web connection
			boolean shouldFormatUrl = !string.toLowerCase().startsWith(HTTP_PROTOCOL)
					&& !string.toLowerCase().startsWith(HTTPS_PROTOCOL);
			String resultString = string;
			if (shouldFormatUrl) {
				if (!resultString.toLowerCase().startsWith(WWW_SUBDOMAIN)) {
					resultString = WWW_SUBDOMAIN + resultString;
				}
				resultString = HTTPS_PROTOCOL + resultString;
			}

			try {
				URL item = new URL(resultString);
				LPLog.INSTANCE.i(TAG, "Returned URL: " + item);
				url = item.toString();
			} catch (Exception e) {
				LPLog.INSTANCE.e(TAG, ERR_00000015, "ERROR", e);
			}
		}
		return url;
	}

	/**
	 * This method opens a connection to the final url from mSourceContent object. If response code indicates redirect - open new connection to the provided location.
	 * @return a BufferedReader object related to the last opened connection.
	 * @throws IOException if there are 3 or more equal redirect links. What, efficiently, is redirects infinite loop protection.
	 * */
	private BufferedReader openConnectionWithAdvancedRedirects() throws IOException {
		URL resourceUrl, base, next;
		Map<String, Integer> visited;
		String location;
		String url = mSourceContent.getFinalUrl();
		int times;
		visited = new HashMap<>();
		while (true) {
			if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
				times = visited.compute(url, (key, count) -> count == null ? 1 : count + 1);
			} else {
				//remove it when min SDK will be 24.
				if (visited.containsKey(url)) {
					Integer oldValue = visited.get(url);
					visited.remove(url);
					visited.put(url, ++oldValue);
					times = oldValue;
				} else {
					visited.put(url, 1);
					times = 1;
				}
			}
			if (times > 3)
				throw new IOException("Stuck in redirect loop");

			resourceUrl = new URL(url);
			hc = (HttpURLConnection) resourceUrl.openConnection();
			hc.setConnectTimeout(CONNECT_TIMEOUT_IN_MILLI);
			hc.setReadTimeout(READ_TIMEOUT_IN_MILLI);
			hc.addRequestProperty("Connection", "keep-alive");
			hc.addRequestProperty("User-Agent", System.getProperty("http.agent"));
			hc.addRequestProperty("Accept-Language", Locale.getDefault().getLanguage());
			hc.setInstanceFollowRedirects(false);   // Make the logic below easier to detect redirections

			switch (hc.getResponseCode()) {
				case HttpURLConnection.HTTP_MOVED_PERM:
				case HttpURLConnection.HTTP_MOVED_TEMP:
				case HttpURLConnection.HTTP_SEE_OTHER:
					location = hc.getHeaderField("Location");
					location = URLDecoder.decode(location, "UTF-8");
					base = new URL(url);
					next = new URL(base, location);  // Deal with relative URLs
					url = next.toExternalForm();
					LPLog.INSTANCE.d(TAG, "moving to " + location);
					continue;
				default:
					LPLog.INSTANCE.d(TAG, "response code: " + hc.getResponseCode());
			}

			break;
		}
		// This is IMPORTANT to prevent cases without og: tags just in mobile
		String charset = getCharsetEncoding(hc.getContentType());
		return new BufferedReader(new InputStreamReader(hc.getInputStream(), charset));
	}

	public void parseHtml(String[] params) {
		String url = prepareLink(params[0]);
		if (mSourceContent == null) {
			mSourceContent = new SourceContent();
		}
		if (url == null || url.isEmpty()) {
			mSourceContent.setFinalUrl("");

			// in case we have a valid url
		} else {
			mSourceContent.setFinalUrl((extendedTrim(url)));
			StringBuilder myString = new StringBuilder();

			try {
				hc = null;
				String thisLine;
				long openConnectionTime = System.currentTimeMillis();
				BufferedReader br = openConnectionWithAdvancedRedirects();
				while ((thisLine = br.readLine()) != null) {
					myString.append(thisLine);
				}
				br.close();
				LPLog.INSTANCE.d(TAG, "HttpURLConnection reading finished: " + mSourceContent.getFinalUrl() + ". Execution time: " + (System.currentTimeMillis() - openConnectionTime) + " ms");
				if (TextUtils.isEmpty(myString)) {
					throw new IOException("the url is empty");
				}

				// if the html is with extended tags
				if (myString.indexOf(KEY_HTML_TAG_HEAD_OPEN) == -1) {
					mSourceContent.setHtmlCode(extendedTrim(myString.toString()));
				} else {
					mSourceContent.setHtmlCode(extendedTrim(myString.substring(myString.indexOf(KEY_HTML_TAG_HEAD_OPEN),
							myString.lastIndexOf(KEY_HTML_TAG_HEAD_CLOSE) + KEY_HTML_TAG_HEAD_CLOSE.length())));
				}

				HashMap<String, String> metaTags = getMetaTags(mSourceContent.getHtmlCode());

				// save the meta tags for later use
				mSourceContent.setMetaTags(metaTags);
				mSourceContent.setTitle(metaTags.get(KEY_HTML_TAG_TITLE));
				mSourceContent.setDescription(metaTags.get(KEY_HTML_TAG_DESCRIPTION));
				mSourceContent.setSiteName(metaTags.get(KEY_HTML_TAG_SITE_NAME));

				// if we didn't found a title
				if (mSourceContent.getTitle().equals(KEY_EMPTY)) {
					String matchTitle = pregMatch(mSourceContent.getHtmlCode(), TITLE_PATTERN, 2);
					if (!matchTitle.equals(KEY_EMPTY)) {
						mSourceContent.setTitle((matchTitle));
					}
				}
				mSourceContent.setDescription(mSourceContent.getDescription().replaceAll(SCRIPT_PATTERN, KEY_EMPTY));

				String tagImage = metaTags.get(KEY_HTML_TAG_IMAGE);
				// if we didn't found an image
				if (tagImage != null && !tagImage.equals(KEY_EMPTY)) {
					String imageLink = metaTags.get(KEY_HTML_TAG_IMAGE);
					if (imageLink != null && !imageLink.contains(HTTP_PROTOCOL_SHORT)) {
						imageLink = mSourceContent.getFinalUrl() + imageLink;
					}
					mSourceContent.setImages(imageLink);
				}
				mSourceContent.setSuccess(true);
				LPLog.INSTANCE.d(TAG, "mSourceContent. inited.");
			} catch (Exception e) {
				mSourceContent.setSuccess(false);
				mSourceContent.setUrl(KEY_EMPTY);
				mSourceContent.setHtmlCode(KEY_EMPTY);
				LPLog.INSTANCE.w(TAG, "url error: " + e.getMessage(), e);
			} finally {
				if (hc != null) {
					hc.disconnect();
				}
			}
		}

		// store the parsed data locally
		if (!TextUtils.isEmpty(mSourceContent.getFinalUrl())) {
			String[] finalLinkSet = mSourceContent.getFinalUrl().split("&");
			mSourceContent.setUrl(finalLinkSet[0]);
			mSourceContent.setCanonicalUrl(canonicalPage(mSourceContent.getFinalUrl()));
			mSourceContent.setDescription((mSourceContent.getDescription()));
			mSourceContent.setSiteName((mSourceContent.getSiteName()));
		}
	}

	@NonNull
	private String getCharsetEncoding(String contentType) {

		String charset = "";

		if (!TextUtils.isEmpty(contentType)) {
			String[] values = contentType.split(";");

			for (String value : values) {
				value = value.trim();

				if (value.toLowerCase().startsWith("charset=")) {
					charset = value.substring("charset=".length());
				}
			}
		}

		if ("".equals(charset)) {
            charset = "UTF-8";
        }
		return charset;
	}

	/**
	 * Returns the canonical url
	 */
	private String canonicalPage(String url) {
		String canonical = "";
		if (url.startsWith(HTTP_PROTOCOL)) {
			url = url.substring(HTTP_PROTOCOL.length());
		} else if (url.startsWith(HTTPS_PROTOCOL)) {
			url = url.substring(HTTPS_PROTOCOL.length());
		}

		int urlLength = url.length();
		for (int i = 0; i < urlLength; i++) {
			if (url.charAt(i) != '/') {
				canonical += url.charAt(i);
			} else {
				break;
			}
		}
		return canonical;
	}

	/**
	 * getMetaTags - extract the 4 tags we are using (image, url, title, description)
	 * using regex for open graph and non open graph tags
	 *
	 * @return Returns meta tags from html code
	 */

	private HashMap<String, String> getMetaTags(String content) {
		HashMap<String, String> metaTags = new HashMap<>();
		metaTags.put(KEY_HTML_TAG_URL, KEY_EMPTY);
		metaTags.put(KEY_HTML_TAG_TITLE, KEY_EMPTY);
		metaTags.put(KEY_HTML_TAG_DESCRIPTION, KEY_EMPTY);
		metaTags.put(KEY_HTML_TAG_IMAGE, KEY_EMPTY);
		metaTags.put(KEY_HTML_TAG_SITE_NAME, KEY_EMPTY);

		List<String> matches = pregMatchMetaTagPattern(content);
		for (String match : matches) {
			final String lowerCase = match.toLowerCase();

			if (Configuration.getBoolean(R.bool.link_preview_to_use_more_than_og_tags)) {
				if (lowerCase.contains("name=\"url\"")
						|| lowerCase.contains("name='url'"))
					updateMetaTag(metaTags, "url", separateMetaTagsContent(match));
				else if (lowerCase.contains("name=\"title\"")
						|| lowerCase.contains("name='title'"))
					updateMetaTag(metaTags, "title", separateMetaTagsContent(match));
				else if (lowerCase.contains("name=\"description\"")
						|| lowerCase.contains("name='description'"))
					updateMetaTag(metaTags, "description", separateMetaTagsContent(match));
				else if (lowerCase.contains("name=\"image\"")
						|| lowerCase.contains("name='image'")
						|| lowerCase.contains("itemprop=\"image\""))
					updateMetaTag(metaTags, "image", separateMetaTagsContent(match));
				else if (lowerCase.contains("name=\"site_name\"")
						|| lowerCase.contains("name='site_name'"))
					updateMetaTag(metaTags, "site_name", separateMetaTagsContent(match));
			}

			if (lowerCase.contains("property=\"og:url\"")
					|| lowerCase.contains("property='og:url'"))
				updateMetaTag(metaTags, "url", separateMetaTagsContent(match));
			else if (lowerCase.contains("property=\"og:title\"")
					|| lowerCase.contains("property='og:title'"))
				updateMetaTag(metaTags, "title", separateMetaTagsContent(match));
			else if (lowerCase.contains("property=\"og:description\"")
					|| lowerCase.contains("property='og:description'"))
				updateMetaTag(metaTags, "description", separateMetaTagsContent(match));
			else if (lowerCase.contains("property=\"og:image\"")
					|| lowerCase.contains("property='og:image'"))
				updateMetaTag(metaTags, "image", separateMetaTagsContent(match));
			else if (lowerCase.contains("property=\"og:site_name\"")
					|| lowerCase.contains("property='og:site_name'"))
				updateMetaTag(metaTags, "site_name", separateMetaTagsContent(match));


		}
		return metaTags;
	}

	private void updateMetaTag(HashMap<String, String> metaTags, String url, String value) {
		if (value != null && (value.length() > 0)) {
			metaTags.put(url, value);
		}
	}

	/**
	 * Gets content from MetaTag
	 */
	private String separateMetaTagsContent(String content) {
		return pregMatch(content, METATAG_CONTENT_PATTERN, 1);
	}

	/**
	 * Removes extra spaces and trim the string
	 *
	 * @param content - the original string
	 * @return the new content
	 */
	private String extendedTrim(String content) {
		return TextUtils.isEmpty(content) ? "" : content.replaceAll("\\s+", " ").replace("\n", " ").replace("\r", " ").trim();
	}

	@VisibleForTesting
	public SourceContent getSourceContent() {
		return mSourceContent;
	}

	private String pregMatch(String content, String pattern, int index) {
		String match = "";
		Matcher matcher = Pattern.compile(pattern).matcher(content);

		if (matcher.find()) {
			match = matcher.group(index);
		}
		return extendedTrim(match);
	}

	private List<String> pregMatchMetaTagPattern(String content) {
		List<String> matches = new ArrayList<>();
		Matcher matcher = Pattern.compile(METATAG_PATTERN).matcher(content);

		while (matcher.find()) {
			matches.add(extendedTrim(matcher.group(1)));
		}
		return matches;
	}
}
