package com.liveperson.infra.loggos

import com.liveperson.infra.BuildConfig
import com.liveperson.infra.analytics.AnalyticsEvent
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.log.LogLine
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.text.SimpleDateFormat
import java.util.*

class LoggosMessageFactory(private val accountId: String) {

	companion object {
		private const val TAG = "LoggosMessage"

		private const val LOGGOS_CONTEXT = "context"
		private const val LOGGOS_ANDROID_CONTEXT = "AndroidSdk"
		private const val SDK_VERSION = "integrationVersion"
		private const val ACCOUNT_ID = "accountId"

		private const val TIME = "time"
		private const val FILENAME = "filename"
		private const val LOG_LEVEL = "logLevel"
		private const val INFO = "info"
		private const val ERROR = "error"
		private const val ERROR_MESSAGE = "error_message"
		private const val TITLE = "title"
		private const val MESSAGE = "msg"
		private const val UUID = "uid"

		private const val EVENT = "event"
		private const val EVENT_NAME = "event_name"
		private const val EVENT_PROPERTIES = "event_properties"
		private const val PROPERTIES = "properties"
		private const val PROPERTY_NAME = "name"
		private const val PROPERTY_VALUE = "value"


		private const val TIMESTAMP_FORMAT = "yyyy-mm-dd HH:MM:SS,SSS"
	}

	/**
	 * Convert log lines into JSON object format
	 */
	fun convertToJson(logLine: LogLine): JSONObject {
		val jsonObject = JSONObject()
		try {
			jsonObject.put(LOGGOS_CONTEXT, LOGGOS_ANDROID_CONTEXT)
			jsonObject.put(SDK_VERSION, BuildConfig.VERSION_NAME)
			jsonObject.put(ACCOUNT_ID, accountId)

			jsonObject.put(TIME, SimpleDateFormat(TIMESTAMP_FORMAT, Locale.ENGLISH).format(logLine.time))
			jsonObject.put(FILENAME, logLine.tag)
			jsonObject.put(LOG_LEVEL, logLine.level.apiName)
			jsonObject.put(TITLE, logLine.message)
			jsonObject.put(MESSAGE, logLine.toString())
			jsonObject.put(UUID, java.util.UUID.randomUUID().toString())

		} catch (e: JSONException) {
			LPLog.w(TAG, "Exception when serializing Logs to LoggosMessages", e)
		}

		return jsonObject
	}

	/**
	 * Convert analytics data into a JSON object as per Kibana standards and return.
	 */
	fun parseAnalyticsData(userProperties: HashMap<String, Any>, analyticsEvent: AnalyticsEvent): JSONObject {
		val analyticsData = JSONObject()
		var logType: String = INFO

		analyticsData.put(LOGGOS_CONTEXT, LOGGOS_ANDROID_CONTEXT)

		val errorMessage = analyticsEvent.getError()
		if (!errorMessage.isNullOrEmpty()) {
			analyticsData.put(ERROR_MESSAGE, errorMessage)
			logType = ERROR
		}
		analyticsData.put(LOG_LEVEL, logType)

		// Map user properties as JSON key-value pairs
		for (prop in userProperties) {
			analyticsData.put(prop.key, prop.value)
		}
		val analyticsEventsJson = parseAnalyticsEvent(analyticsEvent)

		analyticsData.put(EVENT, analyticsEvent.getEventName())
		analyticsData.put(EVENT_PROPERTIES, analyticsEventsJson)
		return analyticsData
	}

	/**
	 * Parse analytics events into JSON object
	 */
	private fun parseAnalyticsEvent(analyticsEvent: AnalyticsEvent): JSONObject {
		val eventObject = JSONObject()
		try {
			val properties = JSONArray()
			eventObject.put(EVENT_NAME, analyticsEvent.getEventName())


			val eventProperties = analyticsEvent.getEventProperties()
			for (property in eventProperties) {
				val prop = JSONObject()
						.put(PROPERTY_NAME, property.propertyName)
						.put(PROPERTY_VALUE, property.propertyValue)
				properties.put(prop)
			}
			eventObject.put(PROPERTIES, properties)
		} catch (exception: Exception) {
			LPLog.w(TAG, "Exception when serializing LPEvents to LoggosMessages", exception)
		}
		return eventObject
	}
}
