package com.liveperson.infra.loggos

import com.liveperson.infra.InternetConnectionService
import com.liveperson.infra.analytics.AnalyticsEvent
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.log.LPLog.getLogSnapshot
import com.liveperson.infra.log.LPLog.i
import com.liveperson.infra.log.LPLog.v
import com.liveperson.infra.log.LPLog.w
import com.liveperson.infra.log.LogLevel
import com.liveperson.infra.managers.PreferenceManager
import com.liveperson.infra.utils.AndroidFrameworkUtils
import com.liveperson.infra.utils.LocalBroadcast
import com.liveperson.infra.utils.unless
import org.jetbrains.annotations.TestOnly
import org.json.JSONObject
import java.util.*
import java.util.concurrent.LinkedBlockingQueue
import javax.net.ssl.SSLPeerUnverifiedException
import kotlin.collections.ArrayList
import kotlin.collections.HashMap

/**
 * Class that manages uploading logs to Loggos.
 */
open class Loggos {

	companion object {
		private const val TAG = "Loggos"

		private const val DEFAULT_BRAND_ID_FOR_PREFS = "default_brand"

		private const val PREFS_KEY_LOGGOS_DOMAIN = "prefs_key_loggos_domain"
		private const val PREFS_KEY_LOGGOS_BRAND_ID = "prefs_key_loggos_targetid"

		private const val CERTIFICATE_ERROR_ACTION = "certificate_error_action"

		private const val NUM_OF_MILLISECONDS_BETWEEN_ERRORS = 100L // 1/10th of a Second
	}

	// Loggos has blocked requests from Mobile SDK. Do not send any logs to Loggos.
	// Re-work on this class once a new logging framework is in place.
	private var enabled = false

	private lateinit var logSender: LoggosUploader
	private lateinit var prefsMgr: PreferenceManager
	private lateinit var internetCheck: () -> Boolean

	private var brandId: String = ""
	private var domain: String = ""
	private var certificates: List<String> = Collections.emptyList()

	private var mLastTimeErrorSent: Long = 0

	/**
	 * Prepares Loggos for uploading
	 * @param brandId The current Brand's Account ID
	 * @param domain The domain we should upload reports to
	 * @param certificates (Optional) A list of Certificate Pinning certs to use.
	 * (Defaults to an empty list.)
	 */
	fun init(brandId: String, domain: String, certificates: List<String> = Collections.emptyList()) {
		this.brandId = brandId
		this.domain = domain
		this.certificates = certificates

		initializeDependencies()

		// Maybe overwrite this.domain and/or this.brandId if either of those are empty
		validateDomainAndBrand()
		// Store the final value of those two variables in preferences
		d(TAG, "Saving Loggos domain (${this.domain}) and brandId (${this.brandId}) to Prefs.")
		prefsMgr.setStringValue(PREFS_KEY_LOGGOS_DOMAIN, DEFAULT_BRAND_ID_FOR_PREFS, this.domain)
		prefsMgr.setStringValue(PREFS_KEY_LOGGOS_BRAND_ID, DEFAULT_BRAND_ID_FOR_PREFS, this.brandId)
		i(TAG, "Loggos initialized successfully.")
	}

	/**
	 * Initialize supporting objects for this class If they are not already done.
	 */
	private fun initializeDependencies() {
		unless (::logSender.isInitialized) {
			logSender = LoggosUploader()
		}
		unless (::prefsMgr.isInitialized) {
			prefsMgr = PreferenceManager.getInstance()
		}
		unless (::internetCheck.isInitialized) {
			internetCheck = { InternetConnectionService.isNetworkAvailable() }
		}
	}

	/**
	 * Set If we want to write logs to Loggos.
	 * This is false If we are running in test environment.
	 * Only useful for unit tests; do not use in Prod.
	 *
	 * @param enabled **true** if you want to send logs to Loggos; **false** otherwise.
	 */
	fun setEnabled(enabled: Boolean) {
		this.enabled = enabled
	}

	/**
	 * Sets a custom Uploader. Only useful for unit tests; do not use in Prod.
	 */
	@TestOnly
	fun setLoggosUploader(loggosUploader: LoggosUploader) {
		logSender = loggosUploader
	}

	/**
	 * Sets a custom internet check function. Only useful for unit tests; do not use in Prod.
	 */
	@TestOnly
	fun setInternetChecker(netCheck: () -> Boolean) {
		internetCheck = netCheck
	}

	/**
	 * Sets a custom Shared Preferences manager.
	 * Only useful for unit tests; do not use in Prod.
	 */
	@TestOnly
	fun setSharedPrefsManager(preferencesManager: PreferenceManager) {
		prefsMgr = preferencesManager
	}

	/**
	 * Reports usage of File Sharing to Loggos
	 */
	fun reportFeatureStatistics() {
		if (enabled) {
			val factory = LoggosMessageFactory(brandId)
			i(TAG, "Uploading feature use report to Loggos...")
			uploadNow(factory)
		}
	}

	/**
	 * Upload all the analytics data collected during the user's interaction with SDK
	 */
	open fun uploadAnalyticsReport(userProperties: HashMap<String, Any>, analyticsEvents: LinkedBlockingQueue<AnalyticsEvent>, callback: LoggosUploader.Callback) {
		initializeDependencies()
		if (isAnalyticsEnabled()) {
			validateDomainAndBrand()
			if (domain.isEmpty() || brandId.isEmpty()) {
				w(TAG, "uploadAnalyticsReport: Cannot upload; Loggos is not initialized.")
				callback.onError(null, Throwable("Loggos is not initialized."))
				return
			}
			val factory = LoggosMessageFactory(brandId)
			val analyticsData = ArrayList<JSONObject>()

			analyticsEvents.mapTo(analyticsData) { analyticsEvent ->
				factory.parseAnalyticsData(userProperties, analyticsEvent)
			}

			i(TAG, "uploadAnalyticsReport: Uploading analytics report to Loggos. Total event: " + analyticsData.size)
			logSender.sendBulk(domain, analyticsData, certificates, object : LoggosUploader.Callback {
				override fun onSuccess(logId: String?) {
					callback.onSuccess(logId)
				}

				override fun onError(messages: List<JSONObject>?, exception: Throwable?) {
					if (exception is SSLPeerUnverifiedException) {
						LocalBroadcast.sendBroadcast(CERTIFICATE_ERROR_ACTION)
					}
					callback.onError(messages, exception)
				}
			})
		} else {
			i(TAG, "Analytics is disabled. Do not send data to loggos")
		}
	}

	fun reportError() {
		if (enabled && shouldSendError()) {
			val factory = LoggosMessageFactory(brandId)
			i(TAG, "Uploading error report to Loggos...")
			uploadNow(factory)

			// Store the last-error-sent time
			mLastTimeErrorSent = System.currentTimeMillis()
		}
	}

	private fun uploadNow(factory: LoggosMessageFactory) { // If we're not initialized yet or there is no network we can't do anything
		// Verify our data is OK
		if (domain.isEmpty() || brandId.isEmpty()) {
			w(TAG, "Cannot upload; Loggos is not initialized.")
			return
		}
		// Verify our internet is OK
		if (!internetCheck()) {
			w(TAG, "Cannot upload; no internet.")
			return
		}

		// Construct the list of JSON Objects
		val messages = ArrayList<JSONObject>()
		getLogSnapshot(LogLevel.VERBOSE).mapTo(messages) {
			logLine -> factory.convertToJson(logLine)
		}

		// Upload to Loggos
		logSender.sendBulk(domain, messages, certificates, object : LoggosUploader.Callback {
			override fun onSuccess(logId: String?) {
				/* No-Op */
			}

			override fun onError(messages: List<JSONObject>?, exception: Throwable?) {
				if (exception is SSLPeerUnverifiedException) {
					LocalBroadcast.sendBroadcast(CERTIFICATE_ERROR_ACTION)
				}
				w(TAG, "uploadNow: Failed to upload error report to loggos: $exception")
			}

		})
	}

	/**
	 * Validate we have domain and brandId, and if not, get them from Preferences
	 */
	private fun validateDomainAndBrand() {
		if (domain.isEmpty()) { // Get from preferences
			domain = prefsMgr.getStringValue(PREFS_KEY_LOGGOS_DOMAIN, DEFAULT_BRAND_ID_FOR_PREFS, "")
			v(TAG, "validateDomainAndBrand: Got domain from preferences: $domain")
		}
		if (brandId.isEmpty()) { // Get from preferences
			brandId = prefsMgr.getStringValue(PREFS_KEY_LOGGOS_BRAND_ID, DEFAULT_BRAND_ID_FOR_PREFS, "")
			v(TAG, "validateDomainAndBrand: Got targetId from preferences: $brandId")
		}
	}

	/**
	 * Checks if we've recently uploaded error statistics to Loggos
	 */
	private fun shouldSendError(): Boolean {
		val needToSend = System.currentTimeMillis() - mLastTimeErrorSent > NUM_OF_MILLISECONDS_BETWEEN_ERRORS
		if (!needToSend) {
			d(TAG, "Time since last error is less than $NUM_OF_MILLISECONDS_BETWEEN_ERRORS millis; no need to send to loggos")
		}
		return needToSend
	}

	/**
	 * Check If we are not running test cases and analytics flag is enabled in site settings.
	 */
	private fun isAnalyticsEnabled(): Boolean {
		return enabled && prefsMgr.getBooleanValue(PreferenceManager.SITE_SETTINGS_SDK_ANALYTICS_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, true)
	}
}
