package com.liveperson.infra.database;


import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import androidx.annotation.NonNull;

import com.liveperson.infra.Clearable;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.database.tables.BaseTable;
import com.liveperson.infra.database.tables.BrandProfileTable;
import com.liveperson.infra.database.tables.ConversationsTable;
import com.liveperson.infra.database.tables.DialogsTable;
import com.liveperson.infra.database.tables.FilesTable;
import com.liveperson.infra.database.tables.MessagesTable;
import com.liveperson.infra.database.tables.UsersTable;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by ofira on 10/07/2015.
 * Modified by nirni on 08/11/2015.
 */
public class DatabaseManager implements ShutDown, Clearable {

    private static final String TAG = "DatabaseManager";
    private static final String DATABASE_NAME = "lp_infra_tables.db";
    // Database Version
    private static final int DATABASE_VERSION = 7;

    private final DatabaseHelper mDBHelper;
    private static DatabaseManager mInstance = null;


    /**
     * A list that holds all additional registered tables
     */
    private List<BaseTable> tableList = new ArrayList<>();

    private DatabaseManager() {
        mDBHelper = new DatabaseHelper();
        // Core tables
        MessagesTable mMessagesTable = new MessagesTable();
        DialogsTable mDialogsTable = new DialogsTable();
        ConversationsTable mConversationsTable = new ConversationsTable();
        BrandProfileTable brandProfileTable = new BrandProfileTable();
        UsersTable mUsersTable = new UsersTable();
        FilesTable mFilesTable = new FilesTable();

        registerTable(mMessagesTable);
        registerTable(mDialogsTable);
        registerTable(mConversationsTable);
        registerTable(brandProfileTable);
        registerTable(mUsersTable);
        registerTable(mFilesTable);

		//call getWritableDatabase in order to check if onUpgrade is needed as part of Init DatabaseManager
        //DO NOT DELETE THIS LINE
		mDBHelper.getWritableDatabase();
    }

    public static DatabaseManager getInstance() {
        if (mInstance == null) {
            synchronized (DatabaseManager.class) {
                if (mInstance == null) {
                    mInstance = new DatabaseManager();
                }
            }
        }

        return mInstance;
    }

    DatabaseHelper getDBHelper() {
        return mDBHelper;
    }

    /**
     * Register a table on this database manager. All registered tables participates in the create
     * table process and get notified upon database update
     *
     * @param table the table to registered
     */
    private void registerTable(@NonNull BaseTable table) {
        tableList.add(table);
    }

    @Override
    public void clear() {
        LPLog.INSTANCE.d(TAG, "removing db");
        Infra.instance.getApplicationContext().deleteDatabase(DATABASE_NAME);
        mInstance = null;
    }

    @Override
    public void shutDown() {
        DataBaseExecutor.killAll(new ICallback<Void, Exception>() {
            @Override
            public void onSuccess(Void value) {
                mDBHelper.close();
                LPLog.INSTANCE.i(TAG, "closed db.");
            }

            @Override
            public void onError(Exception exception) {
                mDBHelper.close();
                LPLog.INSTANCE.i(TAG, "closed db.");
            }
        });
    }

    ////////////////// Inner classes /////////////////////

    class DatabaseHelper extends SQLiteOpenHelper {
        private final String TAG = "DatabaseHelper";

        private DatabaseHelper() {
            super(Infra.instance.getApplicationContext(), DATABASE_NAME, null, DATABASE_VERSION);
	        LPLog.INSTANCE.i(TAG, "initializing db...");
        }

        @Override
        public void onCreate(SQLiteDatabase db) {
            // Create any additional registered tables
            for (BaseTable table : tableList) {
                LPLog.INSTANCE.d(TAG, "Creating table " + table.getName());
                db.execSQL(table.getCreateCommand());
            }

        }

        @Override
        public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
            // Update any additional registered tables
			LPLog.INSTANCE.d(TAG, "Upgrading DB oldVersion = " + oldVersion + " newVersion = "+ newVersion);

			for (BaseTable table : tableList) {
                LPLog.INSTANCE.d(TAG, "Upgrading table " + table.getName());
                table.onTableUpgrade(db, oldVersion, newVersion);
            }
        }
    }
}
