package com.liveperson.infra.loggos

import com.liveperson.infra.InternetConnectionService
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.log.LPLog.getLogSnapshot
import com.liveperson.infra.log.LPLog.i
import com.liveperson.infra.log.LPLog.v
import com.liveperson.infra.log.LPLog.w
import com.liveperson.infra.log.LogLevel
import com.liveperson.infra.managers.PreferenceManager
import com.liveperson.infra.utils.LocalBroadcast
import com.liveperson.infra.utils.unless
import org.jetbrains.annotations.TestOnly
import org.json.JSONObject
import java.util.*
import javax.net.ssl.SSLPeerUnverifiedException
import kotlin.collections.ArrayList

/**
 * Class that manages uploading logs to Loggos.
 */
class Loggos {

	companion object {
		private const val TAG = "Loggos"

		private const val DEFAULT_BRAND_ID_FOR_PREFS = "default_brand"

		private const val PREFS_KEY_LOGGOS_DOMAIN = "prefs_key_loggos_domain"
		private const val PREFS_KEY_LOGGOS_BRAND_ID = "prefs_key_loggos_targetid"

		private const val CERTIFICATE_ERROR_ACTION = "certificate_error_action"

		private const val NUM_OF_MILLISECONDS_BETWEEN_ERRORS = 100L // 1/10th of a Second
	}

	private var enabled = true

	private lateinit var logSender: LoggosUploader
	private lateinit var prefsMgr: PreferenceManager
	private lateinit var internetCheck: () -> Boolean

	private var brandId: String = ""
	private var domain: String = ""
	private var certificates: List<String> = Collections.emptyList()

	private var mLastTimeErrorSent: Long = 0

	/**
	 * Prepares Loggos for uploading
	 * @param brandId The current Brand's Account ID
	 * @param domain The domain we should upload reports to
	 * @param certificates (Optional) A list of Certificate Pinning certs to use.
	 * (Defaults to an empty list.)
	 */
	fun init(brandId: String, domain: String, certificates: List<String> = Collections.emptyList()) {
		this.brandId = brandId
		this.domain = domain
		this.certificates = certificates

		unless (::logSender.isInitialized) {
			logSender = LoggosUploader()
		}
		unless (::prefsMgr.isInitialized) {
			prefsMgr = PreferenceManager.getInstance()
		}
		unless (::internetCheck.isInitialized) {
			internetCheck = { InternetConnectionService.isNetworkAvailable() }
		}

		// Set up callbacks
		logSender.callback = object : LoggosUploader.Callback {
			override fun onSuccess() { /* No-Op */ }
			override fun onError(messages: List<JSONObject>, exception: Throwable?) {
				if (exception is SSLPeerUnverifiedException) {
					LocalBroadcast.sendBroadcast(CERTIFICATE_ERROR_ACTION)
				}
			}
		}

		// Maybe overwrite this.domain and/or this.brandId if either of those are empty
		validateDomainAndBrand()
		// Store the final value of those two variables in preferences
		d(TAG, "Saving Loggos domain (${this.domain}) and brandId (${this.brandId}) to Prefs.")
		prefsMgr.setStringValue(PREFS_KEY_LOGGOS_DOMAIN, DEFAULT_BRAND_ID_FOR_PREFS, this.domain)
		prefsMgr.setStringValue(PREFS_KEY_LOGGOS_BRAND_ID, DEFAULT_BRAND_ID_FOR_PREFS, this.brandId)
		i(TAG, "Loggos initialized successfully.")
	}

	/**
	 * Set If we want to write logs to Loggos.
	 * This is false If we are running in test environment.
	 *
	 * @param enabled **true** if you want to send logs to Loggos; **false** otherwise.
	 */
	fun setEnabled(enabled: Boolean) {
		this.enabled = enabled
	}

	/**
	 * Sets a custom Uploader. Only useful for unit tests; do not use in Prod.
	 */
	@TestOnly
	fun setLoggosUploader(loggosUploader: LoggosUploader) {
		logSender = loggosUploader
	}

	/**
	 * Sets a custom internet check function. Only useful for unit tests; do not use in Prod.
	 */
	@TestOnly
	fun setInternetChecker(netCheck: () -> Boolean) {
		internetCheck = netCheck
	}

	/**
	 * Sets a custom Shared Preferences manager.
	 * Only useful for unit tests; do not use in Prod.
	 */
	@TestOnly
	fun setSharedPrefsManager(preferencesManager: PreferenceManager) {
		prefsMgr = preferencesManager
	}

	/**
	 * Reports usage of File Sharing to Loggos
	 */
	fun reportFeatureStatistics() {
		if (enabled) {
			val factory = LoggosMessageFactory(brandId)
			i(TAG, "Uploading feature use report to Loggos...")
			uploadNow(factory)
		}
	}

	fun reportError() {
		if (enabled && shouldSendError()) {
			val factory = LoggosMessageFactory(brandId)
			i(TAG, "Uploading error report to Loggos...")
			uploadNow(factory)

			// Store the last-error-sent time
			mLastTimeErrorSent = System.currentTimeMillis()
		}
	}

	private fun uploadNow(factory: LoggosMessageFactory) { // If we're not initialized yet or there is no network we can't do anything
		// Verify our data is OK
		if (domain.isEmpty() || brandId.isEmpty()) {
			w(TAG, "Cannot upload; Loggos is not initialized.")
			return
		}
		// Verify our internet is OK
		if (!internetCheck()) {
			w(TAG, "Cannot upload; no internet.")
			return
		}

		// Construct the list of JSON Objects
		val messages = ArrayList<JSONObject>()
		getLogSnapshot(LogLevel.VERBOSE).mapTo(messages) {
			logLine -> factory.convertToJson(logLine)
		}

		// Upload to Loggos
		logSender.sendBulk(domain, messages, certificates)
	}

	/**
	 * Validate we have domain and brandId, and if not, get them from Preferences
	 */
	private fun validateDomainAndBrand() {
		if (domain.isEmpty()) { // Get from preferences
			domain = prefsMgr.getStringValue(PREFS_KEY_LOGGOS_DOMAIN, DEFAULT_BRAND_ID_FOR_PREFS, "")
			v(TAG, "validateDomainAndBrand: Got domain from preferences: $domain")
		}
		if (brandId.isEmpty()) { // Get from preferences
			brandId = prefsMgr.getStringValue(PREFS_KEY_LOGGOS_BRAND_ID, DEFAULT_BRAND_ID_FOR_PREFS, "")
			v(TAG, "validateDomainAndBrand: Got targetId from preferences: $brandId")
		}
	}

	/**
	 * Checks if we've recently uploaded error statistics to Loggos
	 */
	private fun shouldSendError(): Boolean {
		val needToSend = System.currentTimeMillis() - mLastTimeErrorSent > NUM_OF_MILLISECONDS_BETWEEN_ERRORS
		if (!needToSend) {
			d(TAG, "Time since last error is less than $NUM_OF_MILLISECONDS_BETWEEN_ERRORS millis; no need to send to loggos")
		}
		return needToSend
	}
}
