package com.liveperson.infra.utils;

import android.content.Context;
import android.net.Uri;
import android.os.ParcelFileDescriptor;
import androidx.annotation.Nullable;

import com.liveperson.infra.log.LPLog;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

public class FileUtils {
    private static final String TAG = "FileUtils";

    private static final String IMAGES_FOLDER = "images/";
    private static final String VOICE_FOLDER = "voice/";
    private static final String DOCUMENTS_FOLDER = "documents/";

    /**
     * Write the contents of internal file to the external file created.
     * @param context application context
     * @param privateDirectory app's private directory from where you need to copy the file
     * @param publicDirectory device's public directory (downloads/drive) where file is being copied to
     * @return
     */
    public static boolean writeFileContent(Context context, File privateDirectory, Uri publicDirectory)
    {
        FileInputStream inputStream = null;
        ParcelFileDescriptor parcelFileDescriptor = null;
        FileOutputStream fileOutputStream = null;
        try {
            inputStream = new FileInputStream(privateDirectory);
            parcelFileDescriptor = context.getContentResolver().openFileDescriptor(publicDirectory, "w");
            fileOutputStream = new FileOutputStream(parcelFileDescriptor.getFileDescriptor());

            byte[] b = new byte[1024 * 5];
            int len;

            while ((len = inputStream.read(b)) > 0) {
                fileOutputStream.write(b, 0, len);
            }
        } catch (Exception e) {
			LPLog.INSTANCE.e(TAG, "Failed to copy file from privateDir to publicDir. ", e);
            return false;
        } finally {
            try {
                if (inputStream != null) {
                    inputStream.close();
                }
                if (fileOutputStream != null) {
                    fileOutputStream.close();
                }
                if (parcelFileDescriptor != null) {
                    parcelFileDescriptor.close();
                }
            } catch (IOException exception) {
				LPLog.INSTANCE.e(TAG, "Failed to close fileOutputStream/parcelFileDescriptor ", exception);
            }
        }
        return true;
    }

    /**
     * Create new directory inside app's private directory and file if does not exists
     * @param context application context
     * @param path location of file to be created
     * @param filename Name of file to be created
     */
    @Nullable
    public static File getFilePath(Context context, String path, String filename) {
        // Define directory path and file path
        File privateDir = new File(context.getApplicationContext().getFilesDir() + path);
        File filePath = new File(privateDir, filename);

        // Create directory for the current brandId
        if (!privateDir.exists()) {
            if (!privateDir.mkdirs()) {
                LPLog.INSTANCE.e(TAG, "getFilePath: folder could not be created");
                return null;
            }
        }
        return filePath;
    }


    /**
     * Delete all downloaded files shared with agent
     */
    public static void deleteAllSharedFiles(Context context) {

        LPLog.INSTANCE.d(TAG,"deleting all shared files with agent");
        deleteFolder(context, IMAGES_FOLDER);
        deleteFolder(context, VOICE_FOLDER);
        deleteFolder(context, DOCUMENTS_FOLDER);
    }

    /**
     * Delete all files stored on the device. This should be called when the user logs out his account
     */
    private static void deleteFolder(Context context, String folderName) {

        File dirToDelete = new File(context.getFilesDir() + "/" + folderName + "/");

        LPLog.INSTANCE.d(TAG, "delete" + folderName + "Folder: Trying to delete the" + folderName + "folder:" + dirToDelete.getAbsolutePath());

        deleteRecursively(dirToDelete);
    }

    /**
     * Recursively delete all files (Images and documents) and folders from the device.
     */
    private static void deleteRecursively(File fileOrDirectory) {

        if (fileOrDirectory.isDirectory()) {
            LPLog.INSTANCE.d(TAG, "deleteRecursive: deleting directory: " + fileOrDirectory.getAbsolutePath());

            for (File child : fileOrDirectory.listFiles()) {
                LPLog.INSTANCE.d(TAG, "deleteRecursive: deleting file: " + child.getAbsolutePath());
                deleteRecursively(child);
            }
        }

        //noinspection ResultOfMethodCallIgnored
        fileOrDirectory.delete();
    }
}
