package com.liveperson.infra.auth;

import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;

import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.utils.Utils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import static com.liveperson.infra.errors.ErrorCode.ERR_0000003D;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000003E;

public class LPAuthenticationParams implements Parcelable {

	private static final String TAG = "LPAuthenticationParams";

	private LPAuthenticationType mType;
	private String mAuthKey;
	private String mHostAppJWT;
	private String mIssuerDisplayName;
	private String mHostAppRedirectUri;
	private boolean performStepUp;
	private List<String> mCertificatePinningKeys = new ArrayList<>();
	private String mCodeVerifier; // PKCE integration
	/**
	 * Constructs an empty LPAuthenticationParams, default type SIGN_UP.
	 */
	public LPAuthenticationParams() {
		// Default should be signup until we EoL it in June.
		this(LPAuthenticationType.SIGN_UP);
	}

	/**
	 * Constructs an empty LPAuthenticationParams of the specified type.
	 *
	 * @param type The LPAuthenticationType corresponding to the method you wish to use
	 *             to authenticate your users.
	 */
	public LPAuthenticationParams(LPAuthenticationType type) {
		mType = type;
	}

	/**
	 * Reconstructor to rebuild an LPAuthenticationParams from a saved Parcel.
	 *
	 * @param in The Parcel to unpack into an LPAuthenticationParams.
	 */
	protected LPAuthenticationParams(Parcel in) {
		mAuthKey = in.readString();
		mHostAppJWT = in.readString();
		mIssuerDisplayName = in.readString();
		performStepUp = in.readByte() != 0;
		mHostAppRedirectUri = in.readString();
		mCertificatePinningKeys = in.createStringArrayList();
		mCodeVerifier = in.readString();
		int i = in.readInt();
		if (i != -1) {
			mType = LPAuthenticationType.values()[i];
		}
	}

	/**
	 * Authentication with Auth Code flow.
	 * <br/><br/>
	 * Valid only for LPAuthenticationType [AUTH]
	 *
	 * @param authKey - Mandatory authentication key you will need to connect.
	 * @return The LPAuthenticationParams you called this method on (Builder Pattern).
	 */
	public LPAuthenticationParams setAuthKey(String authKey) { // AUTH_CODE flow
		mAuthKey = authKey;
		if (!TextUtils.isEmpty(mAuthKey)) {
			mType = LPAuthenticationType.AUTH;
		}
		return this;
	}

	/**
	 * Optional addition for Auth Code flow
	 * <br/><br/>
	 * Valid only for LPAuthenticationType [AUTH]
	 *
	 * @param hostAppRedirectUri - A Uri to direct the end-user back to your app.
	 * @return The LPAuthenticationParams you called this method on (Builder Pattern).
	 */
	@SuppressWarnings("unused") // Client apps may use this; we're not sure.
	public LPAuthenticationParams setHostAppRedirectUri(String hostAppRedirectUri) { // AUTH_CODE flow
		mHostAppRedirectUri = hostAppRedirectUri;
		if (!TextUtils.isEmpty(mHostAppRedirectUri)) {
			mType = LPAuthenticationType.AUTH;
		}
		return this;
	}

	/**
	 * Authentication with JWT (Implicit Flow)
	 * <br/><br/>
	 * Valid only for LPAuthenticationType [AUTH]
	 *
	 * @param hostAppJWT - Mandatory JWT used to identify your user to LivePerson.
	 * @return The LPAuthenticationParams you called this method on (Builder Pattern).
	 */
	public LPAuthenticationParams setHostAppJWT(String hostAppJWT) { // IMPLICIT flow
		this.mHostAppJWT = hostAppJWT;
		if (!TextUtils.isEmpty(mHostAppJWT)) {
			mType = LPAuthenticationType.AUTH;
		}
		return this;
	}

	/**
	 * Optional public keys, used for certificate pinning
	 *
	 * @param publicKey The Public Key of the Certificate you want to Pin.
	 * @return The LPAuthenticationParams you called this method on (Builder Pattern).
	 */
	@SuppressWarnings("UnusedReturnValue") // Normal for Builder pattern method.
	public LPAuthenticationParams addCertificatePinningKey(String publicKey) {
		if (TextUtils.isEmpty(publicKey)) {
			LPLog.INSTANCE.e(TAG, ERR_0000003D, "Certificate key can't be an empty string");
			return this;
		}
		if (!publicKey.startsWith(Utils.SHA256)) {
			publicKey = Utils.SHA256 + publicKey;
		}
		mCertificatePinningKeys.add(publicKey);

		return this;
	}

	public void setPerformStepUp(boolean stepUp) {
		performStepUp = stepUp;
	}

	public static void printSignupDeprecationNotice() {
		LPLog.INSTANCE.e(TAG, ERR_0000003E,
				"\n================================================================================\n" +
						"Auth Type SIGN_UP is [[ DEPRECATED ]]\n\n" +
						"Instead, please use LPAuthenticationType AUTH or UN_AUTH for AuthCode, Implicit,\n" +
						"or Un-Auth flows.\n\n" +
						"WARNING: LPAuthenticationType SIGN_UP will reach end-of-life on 6/30/2020.\n" +
						"Please contact LivePerson through our support channels to learn how to migrate\n" +
						"your app to Authenticated or Unauthenticated messaging.\n" +
						"================================================================================");
	}

	public static final Creator<LPAuthenticationParams> CREATOR = new Creator<LPAuthenticationParams>() {
		@Override
		public LPAuthenticationParams createFromParcel(Parcel in) {
			return new LPAuthenticationParams(in);
		}

		@Override
		public LPAuthenticationParams[] newArray(int size) {
			return new LPAuthenticationParams[size];
		}
	};


	public String getHostAppJWT() {
		return mHostAppJWT;
	}

	public String getIssuerDisplayName() {
		return mIssuerDisplayName;
	}

	public LPAuthenticationParams setIssuerDisplayName(String issuerDisplayName) {
		this.mIssuerDisplayName = issuerDisplayName;
		return this;
	}

	public String getHostAppRedirectUri() {
		return mHostAppRedirectUri;
	}

	public String getAuthKey() {
		return mAuthKey;
	}

	public boolean getPerformStepUp() {
		return performStepUp;
	}

	public LPAuthenticationParams setCodeVerifier(String codeVerifier) {
		this.mCodeVerifier = codeVerifier;
		return this;
	}
	public String getCodeVerifier() {
		return mCodeVerifier;
	}

	public LPAuthenticationType getAuthType() {
		return mType;
	}

	public boolean isAuthenticated() {
		return mType == LPAuthenticationType.AUTH;//!TextUtils.isEmpty(mAuthKey) || !TextUtils.isEmpty(mHostAppJWT) ;
	}

	public List<String> getCertificatePinningKeys() {
		return mCertificatePinningKeys;
	}

	@Override
	public int describeContents() {
		return 0;
	}

	@Override
	public void writeToParcel(Parcel dest, int flags) {
		dest.writeString(mAuthKey);
		dest.writeString(mHostAppJWT);
		dest.writeString(mIssuerDisplayName);
		dest.writeByte((byte) (performStepUp ? 1 : 0));
		dest.writeString(mHostAppRedirectUri);
		dest.writeStringList(mCertificatePinningKeys);
		dest.writeString(mCodeVerifier);
		dest.writeInt(mType.ordinal());
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;
		LPAuthenticationParams params = (LPAuthenticationParams) o;
		return mType == params.mType
				&& Objects.equals(mAuthKey, params.mAuthKey)
				&& Objects.equals(mHostAppJWT, params.mHostAppJWT)
				&& Objects.equals(mIssuerDisplayName, params.mIssuerDisplayName)
				&& Objects.equals(mHostAppRedirectUri, params.mHostAppRedirectUri)
				&& Objects.equals(mCertificatePinningKeys, params.mCertificatePinningKeys)
				&& Objects.equals(mCodeVerifier, params.mCodeVerifier);
	}

	@Override
	public int hashCode() {
		return Objects.hash(
				mType,
				mAuthKey,
				mHostAppJWT,
				mIssuerDisplayName,
				mHostAppRedirectUri,
				mCertificatePinningKeys,
				mCodeVerifier
		);
	}
}
