package com.liveperson.infra.model

import org.json.JSONObject

private const val KEY_BOLD = "bold"
private const val KEY_ITALIC = "italic"
private const val KEY_TEXT_COLOR = "color"
private const val KEY_BACKGROUND_COLOR = "background-color"
private const val KEY_BORDER_COLOR = "border-color"
private const val KEY_BORDER_RADIUS = "border-radius"

private const val NO_VALUE = -1

/**
 * Extension function over json object used to
 * deserialize quick reply style object from json object.
 *
 * @return deserialized quick reply style.
 */
fun JSONObject.deserializeStyleFromJson(): QuickReplyStyle {
    return QuickReplyStyle(
        optBoolean(KEY_BOLD, false),
        optBoolean(KEY_ITALIC, false),
        optString(KEY_TEXT_COLOR),
        optString(KEY_BORDER_COLOR),
        optInt(KEY_BORDER_RADIUS, NO_VALUE).takeUnless { it < 0 },
        optString(KEY_BACKGROUND_COLOR),
    )
}

/**
 * Extension function over quick reply style used to serialize
 * style to SDK readable json file to represent it.
 *
 * @return serialized quick reply style in LP SDK json format.
 */
fun QuickReplyStyle.serializeToJson(): JSONObject {
    return JSONObject()
        .put(KEY_BOLD, bold)
        .put(KEY_ITALIC, italic)
        .putIfNotNull(KEY_TEXT_COLOR, textColor?.takeUnless { it.isEmpty() })
        .putIfNotNull(KEY_BACKGROUND_COLOR, backgroundColor?.takeUnless { it.isEmpty() })
        .putIfNotNull(KEY_BORDER_COLOR, borderColor?.takeUnless { it.isEmpty() })
        .putIfNotNull(KEY_BORDER_RADIUS, borderRadius)
}

/**
 * Extension function over json object used to put provide string value by key, if this
 * value is not null.
 * This method helps to skip null values while serialization of quick replies styles.
 */
private fun JSONObject.putIfNotNull(key: String, value: String?): JSONObject {
    return if (value != null) {
        put(key, value)
    } else {
        this
    }
}

/**
 * Extension function over json object used to put provide int value by key, if this
 * value is not null.
 * This method helps to skip null values while serialization of quick replies styles.
 */
private fun JSONObject.putIfNotNull(key: String, value: Int?): JSONObject {
    return if (value != null) {
        put(key, value)
    } else {
        this
    }
}