package com.liveperson.infra.network.http.requests;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.text.TextUtils;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.body.HttpRequestBody;
import com.liveperson.infra.network.http.body.LPJSONObjectBody;
import com.liveperson.infra.network.http.request.HttpPostRequest;
import com.liveperson.infra.otel.LPTraceType;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

/**
 * Created by Ilya Gazman on 11/26/2015.
 */
public class PushRequest implements Command {

    private static final String TAG = "PushRequest";

    static final String KEY_APP_ID = "appId";
    static final String KEY_USER_ID = "lpid";
    static final String KEY_PLATFORM = "platform";
    private static final String KEY_REGISTRATION_ID = "pnToken";
    private static final String KEY_DEVICE_ID = "deviceId";
    private static final String KEY_TOKEN_ID = "jwt";
    private static final String KEY_OUTBOUND_MESSAGING_SUPPORT = "outboundMessagingSupport";
    private static final String KEY_AUTH_TYPE = "authType";
    // added from v4.9.1/v5.7.1, more info here: MBK-1666, MBK-2036
    private static final String KEY_OUTBOUND_MESSAGING_RICH_CONTENT = "outboundMessagingRichContent";

    @SuppressWarnings("FieldCanBeLocal")
    private String mDeviceId = "12345678"; //TODO: Need to figure out why use this number or fix this.
    private String mGcmRegistrationId;
    String mAppId;
    private String mTokenId;
    String mUserId;
    String mPlatform;
    private String mPusherURL;
    String userAuthType;
    private List<String> mCertificates;
    @Nullable
    protected ICallback<Void, Exception> mCallback;
    private boolean mIsRegisterRequest;

    public PushRequest(String pusherURL, String userId, String appId, String gcmRegistrationId, String tokenId, String platform, String authType, List<String> certificates, boolean isRegisterRequest) {
        mPusherURL = pusherURL;
        mGcmRegistrationId = gcmRegistrationId;
        mUserId = userId;
        mAppId = appId;
        mTokenId = tokenId;
        mPlatform = platform;
        userAuthType = authType;
        mCertificates = certificates;
        mIsRegisterRequest = isRegisterRequest;
    }

    public PushRequest setAppId(String appId) {
        mAppId = appId;
        return this;
    }

    public PushRequest setCallback(ICallback<Void, Exception> callback) {
        mCallback = callback;
        return this;
    }

    @Override
    public void execute() {

        JSONObject pusherJson = getBody();

        LPTraceType type = mIsRegisterRequest ? LPTraceType.PUSHER_REGISTER_REQ : LPTraceType.PUSHER_UNREGISTER_REQ;
        HttpPostRequest httpPostRequest = new HttpPostRequest(mPusherURL, type);
        HttpRequestBody jsonBody = new LPJSONObjectBody(pusherJson);
        httpPostRequest.setBody(jsonBody);
        httpPostRequest.setCertificatePinningKeys(mCertificates);

        LPLog.INSTANCE.i(TAG, "Pusher url " + mPusherURL);
        LPLog.INSTANCE.d(TAG, "pusherJson " + pusherJson.toString());
        LPLog.INSTANCE.d(TAG, "pusherJson body " + jsonBody.toString());

        httpPostRequest.setCallback(new ICallback<String, Exception>() {
            @Override
            public void onSuccess(String json) {
                handleResponse(json);
            }

            @Override
            public void onError(Exception exception) {
                handleError(exception);
            }

        });
        HttpHandler.execute(httpPostRequest);
    }


    public void handleResponse(String json) {
        if (!TextUtils.isEmpty(json)) {
            try {
                JSONObject response = new JSONObject(json);
                String pnToken = response.optString("pnToken");
                //Success is only if we receive pnToken
                if (!TextUtils.isEmpty(pnToken)){
                    LPLog.INSTANCE.d(TAG, "onSuccess " + json);
                    if (mCallback != null) {
                        mCallback.onSuccess(null);
                    }
                }else{
                    //if response contains "message" - apparently it's error message
                    String errorMsg = response.optString("message");
                    if (!TextUtils.isEmpty(errorMsg)){
                        handleError(new Exception(errorMsg));
                    }else{
                        //empty response
                        handleError(new Exception("Empty register pusher response "+ json));
                    }
                }

            } catch (JSONException e) {
                handleError(new Exception(json));
            }
        }else{
            handleError(new Exception("Empty response!"));
        }
    }

    public void handleError(Exception exception) {
        if (exception != null) {
            LPLog.INSTANCE.d(TAG, "onError ", exception);
            if (mCallback != null) {
                mCallback.onError(new Exception("Failed to register to pusher. response message = ", exception));
            }
        }else{
            if (mCallback != null) {
                mCallback.onError(new Exception("Failed to register to pusher."));
            }
        }
    }
    @NonNull
    protected JSONObject getBody() {

        JSONObject pusherJson = new JSONObject();

        try {
            pusherJson.put(KEY_APP_ID, mAppId);
            pusherJson.put(KEY_USER_ID, mUserId);
            pusherJson.put(KEY_PLATFORM, mPlatform);
            pusherJson.put(KEY_REGISTRATION_ID, mGcmRegistrationId);
            pusherJson.put(KEY_DEVICE_ID, mDeviceId);
            pusherJson.put(KEY_TOKEN_ID, mTokenId);
            pusherJson.put(KEY_AUTH_TYPE, userAuthType);
            pusherJson.put(KEY_OUTBOUND_MESSAGING_SUPPORT, true);
            pusherJson.put(KEY_OUTBOUND_MESSAGING_RICH_CONTENT, true);
        } catch (JSONException ignored) {
        }

        return pusherJson;
    }
}
