package com.liveperson.infra;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;

import androidx.localbroadcastmanager.content.LocalBroadcastManager;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Created by ofira on 16/11/2015.
 */
public class LocalBroadcastReceiver {

    private static final Set<LPBroadcastReceiver> registeredReceivers = new HashSet<>();

    /**
     * Method used to unregister broadcast receiver on logout event to prevent
     * context holding and memory leaks. This method clears all registered receivers
     * within sdk (UMS queries, conversation events etc).
     *
     * @see Infra#clear()
     */
    public static void unregisterAll() {
        synchronized (registeredReceivers) {
            Iterator<LPBroadcastReceiver> receiverIterator = registeredReceivers.iterator();
            while (receiverIterator.hasNext()) {
                LPBroadcastReceiver receiver = receiverIterator.next();
                if (receiver != null) {
                    receiver.unregisterReceiver();
                }
                receiverIterator.remove();
            }
        }
    }

    private final IOnReceive mReceiver;
    private final LPBroadcastReceiver mLocalBroadcastManager;
    private final IntentFilter mIntentFilter;
    private boolean isRegistered;

    /**
     * Unregister a previously registered BroadcastReceiver.
     * All filters that have been registered for this BroadcastReceiver will be
     * removed.
     */
    public void unregister() {
        isRegistered = false;
        mLocalBroadcastManager.unregister();
    }

    /**
     * Register a receive for the local broadcasts
     */
    public void register() {
        isRegistered = true;
        mLocalBroadcastManager.register();
    }

    public interface IOnReceive {
        void onBroadcastReceived(Context context, Intent intent);
    }

    private LocalBroadcastReceiver(Builder builder, IOnReceive receiverCallback) {
        mIntentFilter = builder.mIntentFilter;
        mReceiver = receiverCallback;
        mLocalBroadcastManager = new LPBroadcastReceiver();
        register();
    }

    public boolean isRegistered() {
        return isRegistered;
    }

    public static class Builder {

        private IntentFilter mIntentFilter = new IntentFilter();

        public Builder addAction(String action) {
            mIntentFilter.addAction(action);
            return this;
        }

        public LocalBroadcastReceiver build(IOnReceive receiverCallback) {
            return new LocalBroadcastReceiver(this, receiverCallback);
        }
    }

    public class LPBroadcastReceiver extends BroadcastReceiver {
        private LocalBroadcastManager broadcastManager;

        public LPBroadcastReceiver() {
            broadcastManager = LocalBroadcastManager.getInstance(Infra.instance.getApplicationContext());
        }

        public void register() {
            synchronized (registeredReceivers) {
                registeredReceivers.add(this);
            }
            broadcastManager.registerReceiver(this, mIntentFilter);
        }


        @Override
        public void onReceive(Context context, Intent intent) {
            mReceiver.onBroadcastReceived(context, intent);
        }

        public void unregister() {
            synchronized (registeredReceivers) {
                registeredReceivers.remove(this);
            }
            broadcastManager.unregisterReceiver(this);
        }

        void unregisterReceiver() {
            broadcastManager.unregisterReceiver(this);
        }
    }
}
