package com.liveperson.infra.otel.exporters

import com.liveperson.infra.ICallback
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.network.http.HttpHandler
import com.liveperson.infra.network.http.body.HttpRequestBody
import com.liveperson.infra.network.http.request.HttpPostRequest
import com.liveperson.infra.otel.LPTraceType
import com.liveperson.infra.otel.OtelUtils
import com.liveperson.infra.otel.models.OtlpResource
import com.liveperson.infra.otel.models.OtlpSpan
import com.nimbusds.jose.shaded.gson.Gson
import com.nimbusds.jose.shaded.gson.JsonObject
import java.lang.Exception

class OtlpHttpExporter(endpoint: String, resource: OtlpResource) : OtlpExporter {

    companion object {
        private const val TAG = "OtlpHttpExporter"
        private const val OTLP_EXPORTER_REQUEST_TIMEOUT = 30000
    }
    private var mEndpoint: String = endpoint
    private var mResource: OtlpResource = resource

    override fun export(spans: List<OtlpSpan>) {
        val traceSpan = OtelUtils.createTrace(mResource, spans)
        val gson = Gson()
        val jsonObject: JsonObject? = gson.toJsonTree(traceSpan).asJsonObject
        if (jsonObject != null) {
            d(TAG, "Otel export body: $jsonObject")
            sendPostRequest(getHttpPostRequest(jsonObject))
        }
    }

    private fun sendPostRequest(httpPostRequest: HttpPostRequest) {
        d(TAG, "Otel sendPostRequest: url : " + httpPostRequest.url)
        httpPostRequest.timeout = OTLP_EXPORTER_REQUEST_TIMEOUT
        // Execute the request
        HttpHandler.execute(httpPostRequest)
    }

    private fun getHttpPostRequest(requestBody: JsonObject): HttpPostRequest {
        val httpPostRequest = HttpPostRequest(mEndpoint, LPTraceType.NOT_TRACKING)
        val body: HttpRequestBody = OtlpJsonObjectBody(requestBody)
        httpPostRequest.setBody(body)
        addHeaders(httpPostRequest)

        httpPostRequest.callback = object : ICallback<String, Exception> {
            override fun onSuccess(value: String?) {
                d(TAG, "onSuccess")
            }

            override fun onError(exception: Exception?) {
                d(TAG, "onError")
            }

        }
        return httpPostRequest
    }

    private fun addHeaders(httpPostRequest: HttpPostRequest) {
        httpPostRequest.addHeader("Content-type", "application/json")
        httpPostRequest.addHeader("Accept", "application/json")
    }
}