package com.liveperson.infra.handler;


import android.app.Notification;
import android.app.Notification.Builder;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.graphics.BitmapFactory;
import android.os.Build;
import androidx.annotation.RequiresApi;
import androidx.annotation.VisibleForTesting;

import com.liveperson.infra.R;

/**
 * Created by Ilya Gazman on 11/26/2015.
 */
public class NotificationHandler {

    private static final int NOTIFICATION_ID = 1212322;

    private final String mTitle;
    private final String mMessage;
    private final Context mContext;

    private int mIconResourceId;
    private PendingIntent mPendingIntent;

    public NotificationHandler(Context context, String title, String message) {
        mContext = context;
        mTitle = title;
        mMessage = message;
    }

    /**
     * Set the icon resource for the notification
     *
     * @param iconResourceId Resource id of icon.
     */
    public NotificationHandler setIconResourceId(int iconResourceId) {
        mIconResourceId = iconResourceId;
        return this;
    }

    /**
     * Set the pending intent for the notification
     */
    public NotificationHandler setPendingIntent(PendingIntent pendingIntent) {
        mPendingIntent = pendingIntent;
        return this;
    }

    /**
     * Show notification
     */
    public void showNotification() {
        Notification notification = createPushNotificationBuilder().build();
        Android.getNotificationManager(mContext).notify(NOTIFICATION_ID, notification);
    }

    private Builder createPushNotificationBuilder() {
        String channelId = mContext.getString(R.string.push_notification_channel_id);
        String channelName = mContext.getString(R.string.push_notification_channel_name);
        Builder builder = NotificationHandler.Android.createNotificationBuilder(mContext, channelId, channelName);
        builder.setContentTitle(mTitle)
                .setAutoCancel(true)
                .setDefaults(Notification.DEFAULT_SOUND | Notification.DEFAULT_LIGHTS)
                .setSmallIcon(mIconResourceId)
                .setContentText(mMessage)
                .setCategory(Notification.CATEGORY_MESSAGE)
                .setPriority(Notification.PRIORITY_HIGH);

        return builder;
    }

    public Builder createForegroundServiceNotificationBuilder() {
        String channelId = mContext.getString(R.string.foreground_service_notification_channel_id);
        String channelName = mContext.getString(R.string.foreground_service_notification_channel_name);

        Builder builder = Android.createNotificationBuilder(mContext, channelId, channelName);
        builder = builder
                .setContentTitle(mTitle)
                .setSmallIcon(mIconResourceId)
                .setLargeIcon(BitmapFactory.decodeResource(mContext.getResources(), mIconResourceId))
                .setContentIntent(mPendingIntent)
                .setProgress(0, 0, true);

        return builder;
    }

    public static class Android {

        public static void hideNotification(Context context, String brandId) {
            ((NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE)).cancel(brandId, NOTIFICATION_ID);
        }

        public static Notification.Builder createNotificationBuilder(Context context, String channelId, String channelName) {
            Notification.Builder builder;
            if (Build.VERSION.SDK_INT < android.os.Build.VERSION_CODES.O) {
                builder = new Notification.Builder(context);
            } else {
                createNotificationChannel(context, channelId, channelName);
                builder = new Notification.Builder(context, channelId);
            }

            return builder;
        }

        @RequiresApi(api = Build.VERSION_CODES.O)
        @VisibleForTesting
        public static void createNotificationChannel(Context context, String channelId, String channelName) {
            NotificationChannel notificationChannel = new NotificationChannel(channelId, channelName, NotificationManager.IMPORTANCE_DEFAULT);
            getNotificationManager(context).createNotificationChannel(notificationChannel);
        }

        @VisibleForTesting
        public static NotificationManager getNotificationManager(Context context) {
            return (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
        }
    }
}
