package com.liveperson.infra.model;

import android.os.Parcel;
import android.os.Parcelable;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import android.text.TextUtils;

import com.liveperson.infra.log.LPLog;

import org.json.JSONArray;
import org.json.JSONException;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000043;

/**
 * Created by Yanda on 04/23/2019.
 * <p>
 * Class for quick reply item.
 */
public final class MessageOption implements Parcelable {

    private static final String TAG = "MessageOption";

    private String displayText;
    private String value;
    private JSONArray metadata;
    private QuickReplyStyle style;

    /**
     * @param displayText Text displayed in the quick reply button.
     * @param value       Text send to agent after select the quick reply.
     */
    public MessageOption(@NonNull String displayText, @NonNull String value) {
        this.displayText = displayText;
        if (value.trim().length() == 0) {
            this.value = displayText;
        } else {
            this.value = value;
        }
    }

    /**
     * This constructor might be used by customers.
     */
    @SuppressWarnings("unused")
    public MessageOption(@NonNull String displayText, @NonNull String value, JSONArray metadata) {
        this(displayText, value);
        this.metadata = metadata;
    }

    public MessageOption(@NonNull String displayText, @NonNull String value, JSONArray metadata, QuickReplyStyle style) {
        this(displayText, value);
        this.metadata = metadata;
        this.style = style;
    }

    public String getDisplayText() {
        return displayText;
    }

    public String getValue() {
        return value;
    }

    public JSONArray getMetadata() {
        return metadata;
    }

    private MessageOption(Parcel in) {
        displayText = in.readString();
        value = in.readString();
        String metadataString = in.readString();
        if (!TextUtils.isEmpty(metadataString)) {
            try {
                metadata = new JSONArray(metadataString);
            } catch (JSONException e) {
                LPLog.INSTANCE.e(TAG, ERR_00000043, "Failed to parse JSON.", e);
            }
        }
        style = in.readParcelable(QuickReplyStyle.class.getClassLoader());
    }

    public static final Creator CREATOR = new Creator() {
        @Override
        public MessageOption createFromParcel(Parcel in) {
            return new MessageOption(in);
        }

        @Override
        public MessageOption[] newArray(int size) {
            return new MessageOption[size];
        }
    };

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(displayText);
        dest.writeString(value);
        dest.writeString(metadata == null ? "" : metadata.toString());
        dest.writeParcelable(style, flags);
    }

    @Nullable
    public QuickReplyStyle getStyle() {
        return style;
    }

    public void setStyle(QuickReplyStyle style) {
        this.style = style;
    }
}
