package com.liveperson.infra.model

import android.os.Parcel
import android.os.Parcelable

/**
 * Class which represents a quick reply option style
 *
 * @param bold flag used to apply bold style to option's text
 * @param italic flag used to apply italic style to option's text
 * @param textColor optional string value of text color. Should be in 6-digits hex format, started with #.
 * @param borderColor optional string value of border color. Should be in 6-digits hex format, started with #.
 * @param borderRadius optional value for border radius. Should be bigger than 0.
 * @param backgroundColor optional string value of background color. Should be in 6-digits hex format, started with #.
 */
data class QuickReplyStyle(
    val bold: Boolean,
    val italic: Boolean,
    val textColor: String?,
    val borderColor: String?,
    val borderRadius: Int?,
    val backgroundColor: String?
): Parcelable {
    companion object CREATOR : Parcelable.Creator<QuickReplyStyle> {

        private const val HEX_FORMATTER = "#%06X"

        override fun createFromParcel(parcel: Parcel): QuickReplyStyle {
            return QuickReplyStyle(parcel)
        }

        override fun newArray(size: Int): Array<QuickReplyStyle?> {
            return arrayOfNulls(size)
        }
    }

    constructor(parcel: Parcel) : this(
        parcel.readByte() != 0.toByte(),
        parcel.readByte() != 0.toByte(),
        parcel.readString(),
        parcel.readString(),
        parcel.readInt().takeUnless { it < 0 },
        parcel.readString()
    )

    /**
     * Public constructor used to provide android colors received from resources.
     * This constructor simplifies creation of quick reply style by providing
     * colors from resource to style directly. All colors will be converted
     * to hex string LP SDK format of colors.
     */
    constructor(
        bold: Boolean = false,
        italic: Boolean = false,
        textColor: Int? = null,
        borderColor: Int? = null,
        borderRadius: Int? = null,
        backgroundColor: Int? = null
    ) : this(
        bold,
        italic,
        textColor?.let { String.format(HEX_FORMATTER, it) },
        borderColor?.let { String.format(HEX_FORMATTER, it) },
        borderRadius,
        backgroundColor?.let { String.format(HEX_FORMATTER, it) },
    )

    override fun writeToParcel(parcel: Parcel, flags: Int) {
        parcel.writeByte(if (bold) 1 else 0)
        parcel.writeByte(if (italic) 1 else 0)
        parcel.writeString(textColor)
        parcel.writeString(borderColor)
        parcel.writeInt(borderRadius ?: 0)
        parcel.writeString(backgroundColor)
    }

    override fun describeContents(): Int {
        return 0
    }
}