package com.liveperson.infra.network.http

import okhttp3.Response
import org.json.JSONObject

private const val KEY_CODE = "code"
private const val CODE_JWT_EXPIRED = "0005"

/**
 * Extensions function for [Throwable] to determine whether produced throwable
 * is cause by JWT expiration.
 *
 * @see Throwable
 * @see HttpException
 *
 * @return true if throwable is caused by jwt expiration, false otherwise.
 */
val Throwable.isTokenExpired: Boolean
    get() {
        val exception = this as? HttpException ?: return false
        return exception.takeIf { it.code == 401 }
            ?.let { it.body }
            ?.let { runCatching { JSONObject(it).optString(KEY_CODE) == CODE_JWT_EXPIRED }.getOrNull() }
            ?: false
    }

/**
 * Extension function for [Response] to determine failure of http request.
 *
 * @return [HttpException] if request result code is not in [200..399],
 * or plain [Exception] otherwise.
 */
fun Response?.asException(): Exception? {
    return when {
        this == null -> {
            Exception("On Response Error : no Response")
        }
        code !in 200..399 -> {
            HttpException(code, body?.string())
        }
        else -> {
            null
        }
    }
}