package com.liveperson.infra.network.socket;


import android.text.TextUtils;

import com.liveperson.infra.Infra;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.model.SocketConnectionParams;
import com.liveperson.infra.network.socket.state.SocketStateListener;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;

import java.util.HashMap;
import java.util.Map;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000047;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000049;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000004A;

import androidx.annotation.VisibleForTesting;

/**
 * Created by Ilya Gazman on 11/3/2015.
 */
public class SocketManager implements ShutDown {

    private static final String TAG = "SocketManager";
    private final Object mSyncObject = new Object();
    private final ResponseMap mResponseMap = new ResponseMap();
    private Map<String, SocketHandler> mSocketHandlersMap;

    private SocketHandlerFactory socketHandlerFactory;


    private static volatile SocketManager Instance = null;


    private SocketManager() {
        mSocketHandlersMap = new HashMap<>();
    }


    /**
     * Get instance of the infra Preference Manager
     * Note: this class uses the application context initialized in the {@link Infra} class
     */
    public static SocketManager getInstance() {
        if (Instance == null) {
            synchronized (SocketManager.class) {
                if (Instance == null) {
                    Instance = new SocketManager();
                }
            }
        }
        return Instance;
    }

    /**
     * Will get a new socket and start connection process
     * if socket already exist, it will be used to resume connection
     */
    public void connect(SocketConnectionParams connectionParams) {
        if (TextUtils.isEmpty(connectionParams.getUrl())) {
            LPLog.INSTANCE.e(TAG, ERR_00000047, "Can't connect to empty url");
            return;
        }
        LPLog.INSTANCE.i(TAG, "connecting to socket");
        SocketHandler socketHandler = obtainSocket(connectionParams.getUrl());
        if (socketHandler != null) {
            socketHandler.connect(connectionParams);
        }
    }

    /**
     * Create a new socket and put it in a local map
     * In case we already have existing socket, we return it
     */
    private SocketHandler obtainSocket(final String url) {
        if (socketHandlerFactory == null) {
            socketHandlerFactory = new SocketHandlerFactory();
        }
        if (TextUtils.isEmpty(url)) {
            LPLog.INSTANCE.e(TAG, ERR_00000049, "Can't obtain socket! url is empty");
            return null;
        }
        synchronized (mSyncObject) {
            SocketHandler socketHandler = mSocketHandlersMap.get(url);
            if (socketHandler == null) {
                socketHandler = socketHandlerFactory.createSocketHandler(mResponseMap);//new SocketHandler(mResponseMap);
                mSocketHandlersMap.put(url, socketHandler);
            }
            return socketHandler;
        }
    }

    /**
     * Send request over the socket
     */
    public void send(BaseSocketRequest baseSocketRequest) {
        SocketHandler socketHandler = getSocketHandler(baseSocketRequest);
        if (socketHandler != null) {
            LPLog.INSTANCE.i(TAG, "Sending request " + baseSocketRequest.getRequestName());
            String data = baseSocketRequest.getData();
            //data = data.replace("\\", "");
            socketHandler.send(data);
        }
    }

    /**
     * Send pending messages of active socket
     */
    public void sendPendingMessages(BaseSocketRequest baseSocketRequest) {
        SocketHandler socketHandler = getSocketHandler(baseSocketRequest);
        if (socketHandler != null) {
            socketHandler.sendPendingMessages();
        }
    }

    private SocketHandler getSocketHandler(BaseSocketRequest baseSocketRequest) {
        if (baseSocketRequest == null) {
            LPLog.INSTANCE.e(TAG, ERR_0000004A, "can't get SocketHandler with null request");
            return null;
        }
        LPLog.INSTANCE.d(TAG, "getSocketHandler req id " + baseSocketRequest.getRequestId());
        BaseResponseHandler responseHandler = baseSocketRequest.getResponseHandler();
        if (responseHandler != null) {
            LPLog.INSTANCE.d(TAG, "getResponseHandler for request " + baseSocketRequest.getRequestId() + " is not null");
            responseHandler.init(baseSocketRequest);
            mResponseMap.putRequestIdHandler(baseSocketRequest.getRequestId(), responseHandler);
        }
        return obtainSocket(baseSocketRequest.getSocketUrl());
    }

    /**
     * Disconnecting the socket, anc clear any callbacks or pending messages
     *
     * @param url connection url
     */
    private void killSocket(String url) {
        LPLog.INSTANCE.d(TAG, "kill socket");
        synchronized (mSyncObject) {
            SocketHandler socketHandler = mSocketHandlersMap.get(url);
            if (socketHandler == null) {
                return;
            }
            socketHandler.disconnect();
            socketHandler.dispose();
        }
    }


    /**
     * Disconnecting the socket
     *
     * @param url connection url
     */
    public void disconnect(String url) {
        LPLog.INSTANCE.d(TAG, "disconnect " + url);
        synchronized (mSyncObject) {
            SocketHandler socketHandler = mSocketHandlersMap.get(url);
            if (socketHandler == null) {
                return;
            }
            socketHandler.disconnect();
        }
    }

    @Override
    public void shutDown() {
        LPLog.INSTANCE.i(TAG, "Shutting down all");
        for (String connectionUrl : mSocketHandlersMap.keySet()) {
            killSocket(connectionUrl);
        }
        mSocketHandlersMap.clear();
        mResponseMap.shutDown();
        Instance = null;
    }

    /**
     * Allow you to register to socket callbacks, like error and data.
     * You can call this method right after connect
     *
     * @param url connection url
     */
    public void registerToSocketState(String url, SocketStateListener callback) {
        SocketHandler socketHandler = obtainSocket(url);
        if (socketHandler != null) {
            socketHandler.getSocketStateManager().register(callback);
        }
    }

    /**
     * unregister from socket state changes callbacks
     *
     * @param url connection url
     */
    public void unregisterFromSocketState(String url, SocketStateListener callback) {
        SocketHandler socketHandler;
        synchronized (mSyncObject) {
            socketHandler = mSocketHandlersMap.get(url);
        }
        if (socketHandler != null) {
            socketHandler.getSocketStateManager().unregister(callback);
        }
    }

    /**
     * Get socket state
     *
     * @param url connection url
     * @return the current state of the socket, or INIT in case there is no active socket
     */
    public SocketState getSocketState(String url) {
        synchronized (mSyncObject) {
            SocketHandler socketHandler = mSocketHandlersMap.get(url);
            if (socketHandler == null) {
                return SocketState.INIT;
            }
            SocketStateManager socketStateManager = socketHandler.getSocketStateManager();
            if (socketStateManager == null) {
                return SocketState.INIT;
            }
            return socketStateManager.getState();
        }
    }

    @VisibleForTesting
    public void putGeneralHandlerMap(GeneralResponseHandler generalResponseHandler) {
        mResponseMap.putGeneralHandler(generalResponseHandler);
    }

    @VisibleForTesting
    public SocketHandlerFactory getSocketHandlerFactory() {
        return socketHandlerFactory;
    }

    @VisibleForTesting
    public void setSocketHandlerFactory(SocketHandlerFactory socketHandlerFactory) {
        this.socketHandlerFactory = socketHandlerFactory;
    }

    public static class SocketHandlerFactory {
        public SocketHandler createSocketHandler(ResponseMap mResponseMap) {
            SocketHandler socketHandler = new SocketHandler(mResponseMap);
            socketHandler.init();
            return socketHandler;
        }
    }
}
