package com.liveperson.infra.utils;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;
import android.os.MessageQueue;
import android.os.Process;

import com.liveperson.infra.IDisposable;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.socket.IdleQueueHandler;

import java.lang.ref.WeakReference;
import java.util.concurrent.CountDownLatch;

import static com.liveperson.infra.errors.ErrorCode.*;

/**
 * Created by Ofir Antebi on 11/10/2015.
 */

public class DispatchQueue extends HandlerThread implements MessageQueue.IdleHandler, IDisposable {

    private static final String TAG = "DispatchQueue";

    private volatile Handler mHandler = null;
    private IdleQueueHandler mIdleQueueHandler;
    private MessageQueue mMessageQueue;
    private HandleMessageCallback mHandleMessageCallback;
    private CountDownLatch mSyncLatch = new CountDownLatch(1);


    /**
     * Create a DispatchQueue with default priority(Process.THREAD_PRIORITY_DEFAULT)
     * and without to be notified when the queue is in idle time
     *
     * @param threadName - the thread name
     */
    public DispatchQueue(String threadName) {
        this(threadName, Process.THREAD_PRIORITY_DEFAULT);
    }


    /**
     * Create a DispatchQueue with priority and without to be
     * notified when the queue is in idle time
     *
     * @param threadName     - the thread name
     * @param threadPriority - The priority to run the thread at. The value supplied must be from
     *                       {@link android.os.Process} and not from java.lang.Thread.
     */
    public DispatchQueue(String threadName, int threadPriority) {
        this(threadName, threadPriority, null);
    }


    /**
     * Create a DispatchQueue with default priority {@value Process#THREAD_PRIORITY_DEFAULT}
     * and register to queueIdle callbacks
     *
     * @param threadName - the thread name
     * @param idleQueue  - callback to be called when the queue is idle
     */
    public DispatchQueue(String threadName, IdleQueueHandler idleQueue) {
        this(threadName, Process.THREAD_PRIORITY_DEFAULT, idleQueue);
    }


    /**
     * Create a DispatchQueue with priority and register to queueIdle callbacks
     *
     * @param threadName     - the thread name
     * @param threadPriority - The priority to run the thread at. The value supplied must be from
     *                       {@link android.os.Process} and not from java.lang.Thread.
     * @param idleQueue      - callback to be called when the the queue is idle
     */
    public DispatchQueue(String threadName, int threadPriority, IdleQueueHandler idleQueue) {
        super(threadName, threadPriority);
        // TODO: 14/06/2016 take care of that
        //setDefaultUncaughtExceptionHandler(SDKUncaughtExceptionHandler.getInstance());
        mHandleMessageCallback = new HandleMessageCallback.NullHandleMessageCallback();
        mIdleQueueHandler = idleQueue;
        start();
    }


    @Override
    protected void onLooperPrepared() {
        if (mIdleQueueHandler != null) {
            mMessageQueue = Looper.myQueue();
            mMessageQueue.addIdleHandler(this);
        }
        mHandler = new DispatchQueueHandler(getLooper(), this);
        mSyncLatch.countDown();
    }

    /**
     * Set message handler callback
     * Called when message is arriving to the handle message method
     */
    public void setHandleMessageCallback(final HandleMessageCallback callback) {
        postRunnable(() -> mHandleMessageCallback = callback);
    }

    /**
     * Pushes a message into the end of the dispatch queue after all pending messages
     * before the current time
     *
     * @param msg Message to send.
     */
    public void sendMessage(Message msg) {
        sendMessage(msg, 0);
    }


    /**
     * Pushes a message onto the end of the dispatch queue after all pending messages
     * before the current time and add a delay
     *
     * @param msg   Message to send.
     * @param delay Delay.
     */
    public void sendMessage(Message msg, int delay) {
        try {
            mSyncLatch.await();
            mHandler.sendMessageDelayed(msg, delay);
        } catch (Exception e) {
            LPLog.INSTANCE.e(TAG + getName(), ERR_00000002, "Handler is not ready", e);
        }
    }


    /**
     * Causes the Runnable r to be added to the message queue.
     */
    public void postRunnable(Runnable runnable) {
        postRunnable(runnable, 0);
    }

    /**
     * Causes the Runnable runnable to be added to the message queue, to be run
     * after the specified amount of time elapses.
     */
    public void postRunnable(Runnable runnable, long delay) {
        try {
            mSyncLatch.await();
            mHandler.postDelayed(runnable, delay);
        } catch (Exception e) {
            LPLog.INSTANCE.e(TAG + getName(), ERR_00000003, "Handler is not ready", e);
        }
    }


    /**
     * Remove any pending posts of runnable that are in the message queue.
     */
    public void removeRunnable(Runnable runnable) {
        try {
            mSyncLatch.await();
            mHandler.removeCallbacks(runnable);
        } catch (Exception e) {
            LPLog.INSTANCE.e(TAG + getName(), ERR_00000004, "Handler is not ready", e);
        }
    }

    /**
     * Remove any pending posts of runnable that are in the message queue.
     */
    public boolean removeMessage(int messageWhat) {
        try {
            mSyncLatch.await();
            if (mHandler.hasMessages(messageWhat)){
                mHandler.removeMessages(messageWhat);
                return true;
            }
        } catch (Exception e) {
            LPLog.INSTANCE.e(TAG + getName(), ERR_00000005, "Handler is not ready", e);
        }
        return false;
    }

    /**
     * Remove all callbacks and messages
     */
    public void cleanupQueue() {
        try {
            mSyncLatch.await();
            mHandler.removeCallbacksAndMessages(null);
        } catch (Exception e) {
            LPLog.INSTANCE.e(TAG + getName(), ERR_00000006, "Handler is not ready", e);
        }
    }


    @Override
    public boolean queueIdle() {
        if (mIdleQueueHandler != null) {
            mIdleQueueHandler.queueIdle();
            return true;
        }
        return false;
    }

    @Override
    public void dispose() {
        try {
            mSyncLatch.await();
            if (mMessageQueue != null) {
                mMessageQueue.removeIdleHandler(this);
            }
            mHandler.getLooper().quit();
            quit();
            LPLog.INSTANCE.d(TAG, "dispose " + Thread.currentThread().getName());
        } catch (Exception e) {
            LPLog.INSTANCE.e(TAG + getName(), ERR_00000007, "Handler is not ready", e);
        }
    }

    /**
     * Check if you run on the dispatch thread
     *
     * @return TRUE if you are running on the calling thread, FALSE otherwise
     */
    public boolean isCurrentThread() {
        try {
            mSyncLatch.await();
            return mHandler.getLooper().getThread() == Thread.currentThread();
        } catch (Exception e) {
            LPLog.INSTANCE.e(TAG + " " + getName(), ERR_00000008, "Handler latch problem", e);
        }
        return false;
    }

    public boolean isThreadAlive() {
        return isAlive();
    }

    /**
     *
     */
    private static class DispatchQueueHandler extends Handler {

        private final WeakReference<DispatchQueue> dispatchQueue;

        DispatchQueueHandler(Looper looper, DispatchQueue dispatchQueue) {
            super(looper);
            this.dispatchQueue = new WeakReference<>(dispatchQueue);
        }

        @Override
        public void handleMessage(Message msg) {
            DispatchQueue dispatchQueue = this.dispatchQueue.get();
            if (dispatchQueue != null) {
                dispatchQueue.mHandleMessageCallback.onHandleMessage(msg);
            }
        }
    }
}
