package com.liveperson.infra.utils;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000018;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000019;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000001A;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.ContentResolver;
import android.content.Context;
import android.content.res.TypedArray;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.provider.MediaStore;
import android.text.TextUtils;
import android.util.Base64;
import android.view.WindowManager;

import androidx.annotation.NonNull;
import androidx.exifinterface.media.ExifInterface;

import com.liveperson.infra.Infra;
import com.liveperson.infra.log.LPLog;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.UUID;

/**
 * Created by nirni on 7/3/16.
 */
public class ImageUtils {

	public static final String LP_PICASSO_TAG = "picasso_lp";

	private static final String TAG = "ImageUtils";

	public static final String PNG = "png";
	public static final String JPG = "jpg";
	private static final String SAMSUNG = "samsung";
	public static final String FILE_MIME_TYPE_GIF = "image/gif";
	public static final String FILE_MIME_TYPE_JPG = "image/jpeg";

	private static final String IMAGES_FOLDER = "images/";
	private static final String FULL_IMAGE_FOLDER = "full/";
	private static final String PREVIEW_IMAGE_FOLDER = "preview/";

	public enum ImageFolderType {PREVIEW, FULL}


	public static String bitmapToBase64(byte[] bitmapByteArray) {

		LPLog.INSTANCE.d(TAG, "bitmapToBase64: Bitmap size: " + bitmapByteArray.length / 1000 + " kb");
		return Base64.encodeToString(bitmapByteArray, Base64.DEFAULT);
	}

	/**
	 * Convert base64 string to an image byte array. Since the base64 received is with a prefix (data:image/jpeg;base64,)
	 * and the decoder does not decode the base64 correctly with it, we need to remove it before decoding
	 */
	private static byte[] base64ToByteArray(String base64) {

		if (base64 != null) {
			LPLog.INSTANCE.d(TAG, "base64ToByteArray: converting base64 to byte array");

			// Remove any existing base64 prefix (e.g. data:image/jpeg;base64,)
			base64 = base64.replaceFirst("data.*base64,", "");

			return Base64.decode(base64, Base64.DEFAULT);
		}

		return null;
	}

	public static String bitmapToBase64(Bitmap bitmap) {

		ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
		bitmap.compress(Bitmap.CompressFormat.PNG, 100, byteArrayOutputStream);
		byte[] byteArray = byteArrayOutputStream.toByteArray();

		LPLog.INSTANCE.d(TAG, "bitmapToBase64: Bitmap size: " + byteArray.length / 1000 + " kb");
		return Base64.encodeToString(byteArray, Base64.DEFAULT);
	}

    /**
     * Can't be unit tested. issue: todo: Mock Picasso.
     */
    public static Bitmap createResizedBitmap(Context context, Uri imageUri, int reqLongerSide, int orientation, boolean fromCamera) throws IOException {

		Bitmap bitmap = PicassoUtils.get(context)
				.load(imageUri)
				.tag(LP_PICASSO_TAG)
				.resize(reqLongerSide, reqLongerSide)
				.onlyScaleDown()
				.centerInside()
				.get();

		// When picasso resize the img, The Exif not been saved
		// Create a new bitmap with the original orientation (from the exif)
		Matrix matrix = new Matrix();
		matrix.postRotate(decodeExifOrientation(orientation, fromCamera));

		return Bitmap.createBitmap(bitmap, 0, 0, bitmap.getWidth(), bitmap.getHeight(), matrix, true);
	}

	/**
	 * createBitmapFromURL - generate a bitmap image from a url with specific resize.
	 * Can't be unit tested. issue: Mock Picasso.
	 */
	public static Bitmap createBitmapFromURL(Context context, String imageUrl, int width, int height) {
		if (TextUtils.isEmpty(imageUrl)) {
			return null;
		}
		try {
			return PicassoUtils.get(context).load(imageUrl).tag(LP_PICASSO_TAG).resize(Math.max(width, 250), Math.max(height, 250)).centerCrop().onlyScaleDown().get();

		} catch (Exception e) {
			try {
				return PicassoUtils.get(context).load(imageUrl).tag(LP_PICASSO_TAG).get();
			} catch (Exception e2) {
				return null;
			}
		}
	}

	@NonNull
	public static Bitmap getBitmap(String imagePath) {
		final BitmapFactory.Options options = new BitmapFactory.Options();
		Bitmap bitmap = BitmapFactory.decodeFile(imagePath, options);
		bitmap = Bitmap.createScaledBitmap(bitmap, bitmap.getWidth(), bitmap.getHeight(), true);

		return bitmap;
	}

	public static int getImageRotation(String imagePath, boolean fromCamera) {

		try {
			ExifInterface exif = new ExifInterface(imagePath);
			int orientation = exif.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL);
			return decodeExifOrientation(orientation, fromCamera);
		} catch (IOException e) {
			LPLog.INSTANCE.i(TAG, "getImageRotation: cannot get exif information on image. Return rotation 0");
			return 0;
		}

	}

	/**
	 * This returns the path of the image given by the Uri. If the Uri is with 'content' scheme it gets
	 * the path from the content resolver. If the scheme is 'file' it simply gets the path from the Uri.
	 * Otherwise this method returns null.
	 *
	 * @return the image path, or null if cannot extract it
	 */
	public static String getImagePath(Context context, Uri selectedImageUri) {

		String path = null;
		String scheme = selectedImageUri.getScheme();

        // If scheme is 'content'
        if (scheme != null && scheme.equals(ContentResolver.SCHEME_CONTENT)) {
            String authority = selectedImageUri.getAuthority();
            String packName = context.getPackageName();
            // The prefix here is the same as defined in the provider in the manifest
            if (authority != null && authority.equals(Infra.getFileProviderAuthorityPrefix() + packName)) {
                //file saved in local app folder, most probably when user took a photo with camera
                path = new File(context.getFilesDir(), selectedImageUri.getPath()).getAbsolutePath();
            } else {
                String[] filePathColumn = {MediaStore.Images.Media.DATA};
                Cursor cursor = context.getContentResolver().query(selectedImageUri, filePathColumn, null, null, null);
                if (cursor != null) {
                    cursor.moveToFirst();
                    int columnIndex = cursor.getColumnIndex(filePathColumn[0]);
                    path = cursor.getString(columnIndex);
                    cursor.close();
                    if (TextUtils.isEmpty(path)) {
                        path = selectedImageUri.toString();
                    }
                }
            }

		}
		// If scheme is 'file'
		else if (scheme != null && scheme.equals(ContentResolver.SCHEME_FILE)) {
			LPLog.INSTANCE.d(TAG, "getImagePath: Uri scheme is file. We get the path of the file");
			path = selectedImageUri.getPath();
		}
		// Unknown scheme
		else {
			LPLog.INSTANCE.e(TAG, ERR_00000018, "getImagePath: Unknown scheme: " + scheme + ". Cannot get image path");
		}

		LPLog.INSTANCE.d(TAG, "getImagePath: path = " + path);
		return path;
	}

	/**
	 * This method is used to compress and get a byte array for the given bitmap
	 */
	public static byte[] getOutputStreamFromBitmap(Bitmap bitmapImage, int compression, String imageTypeExtension) {

		byte[] byteArray = null;

		try {

			LPLog.INSTANCE.d(TAG, "getOutputStreamFromBitmap: file extension: " + imageTypeExtension);
			Bitmap.CompressFormat format = Bitmap.CompressFormat.JPEG;
			if (imageTypeExtension.toLowerCase().equals(PNG)) {
				format = Bitmap.CompressFormat.PNG;
			}

			LPLog.INSTANCE.d(TAG, "getOutputStreamFromBitmap: compressing bitmap to byte array");
			ByteArrayOutputStream os = new ByteArrayOutputStream();
			// Use the compress method on the BitMap object to write image to the OutputStream
			bitmapImage.compress(format, compression, os);
			byteArray = os.toByteArray();
			os.close();

		} catch (IOException e) {
			LPLog.INSTANCE.w(TAG, "getOutputStreamFromBitmap: ", e);
		}

		return byteArray;
	}

	/**
	 * Convert the given base64 to image byte array and save it to disk with random generated name
	 */
	public static String saveBase64ToDisk(Context context, String base64, String brandId) {

		if (base64 != null) {

			// Convert the base64 yo byte array
			byte[] byteArray = base64ToByteArray(base64);
			if (byteArray != null) {
				return saveBitmapToDisk(context, byteArray, brandId, ImageFolderType.PREVIEW, null);
			}
		}

		return null;
	}

	/**
	 * Generates a random filename and save it to disk
	 *
	 * @return path to saved file
	 */
	public static String saveBitmapToDisk(Context context, byte[] imageByteArray, String brandId, ImageFolderType imageFolderType, String fileExtension) {

		if (fileExtension == null) {
			fileExtension = ImageUtils.JPG;
		}

		// Generate a random name
		String filename = UUID.randomUUID().toString() + "." + fileExtension;

		return saveBitmapToDisk(context, imageByteArray, brandId, filename, imageFolderType);
	}

	/**
	 * Save the given byte array to disk to location according to imageFolderType
	 *
	 * @return path to saved file
	 */
	private static String saveBitmapToDisk(Context context, byte[] imageByteArray, String brandId, String filename, ImageFolderType imageFolderType) {

		String imageTypeFolder;
		switch (imageFolderType) {
			case PREVIEW:
				imageTypeFolder = PREVIEW_IMAGE_FOLDER;
				break;
			case FULL:
				imageTypeFolder = FULL_IMAGE_FOLDER;
				break;
			default:
				imageTypeFolder = "";
		}

		String path = "/" + IMAGES_FOLDER + brandId + "/" + imageTypeFolder;

		File filePath = FileUtils.getFilePath(context, path, filename);

		if (filePath == null) {
			return null;
		}

		return saveBitmapToDisk(imageByteArray, filePath);
	}

    /**
     * Save the given byte array to disk to the given file
     *
     * @return path to saved file
     */
    public static String saveBitmapToDisk(byte[] imageByteArray, File file) {


        LPLog.INSTANCE.d(TAG, "saveBitmapToDisk: filePath: " + file.getAbsolutePath());

        try (FileOutputStream fos = new FileOutputStream(file)) {
            fos.write(imageByteArray);
        } catch (FileNotFoundException e) {
            LPLog.INSTANCE.e(TAG, ERR_00000019, "saveBitmapToDisk: File not found", e);
            return null;
        } catch (IOException e) {
            LPLog.INSTANCE.e(TAG, ERR_0000001A, "saveBitmapToDisk: IOException", e);
        }

        LPLog.INSTANCE.d(TAG, "saveBitmapToDisk: file absolute path: " + file.getAbsolutePath());
        return file.getAbsolutePath();
    }

	/**
	 * Create image file (used to provide to camera) and return its Uri
	 */
	public static Uri createImageFileForCamera(Context context, String brandId) throws IOException {

		File lp_images_folder = new File(context.getFilesDir(), ImageUtils.IMAGES_FOLDER + brandId + "/" + ImageUtils.FULL_IMAGE_FOLDER);

		//noinspection ResultOfMethodCallIgnored
		lp_images_folder.mkdirs();

        return AndroidFrameworkUtils.getUriForFile(context, lp_images_folder);
    }

	private static int decodeExifOrientation(int orientation, boolean fromCamera) {
		int degrees;
		switch (orientation) {
			case ExifInterface.ORIENTATION_ROTATE_90:
				degrees = 90;
				break;
			case ExifInterface.ORIENTATION_ROTATE_180:
				degrees = 180;
				break;
			case ExifInterface.ORIENTATION_ROTATE_270:
				degrees = 270;
				break;
			default:
				degrees = 0;
				break;
		}

		degrees = handleSpecificDevices(fromCamera, degrees);
		return degrees;
	}

	/**
	 * If required, change the rotation degrees according to device and according to whether the image if from camera or not (gallery)
	 * @param fromCamera true - image from camera, false - from gallery
	 * @param degrees degrees to manipulate
	 * @return the manipulated degrees
	 */
	private static int handleSpecificDevices(boolean fromCamera, int degrees) {

        String manufacturer = AndroidFrameworkUtils.getManufacturer();

		// If the image if from a Samsung gallery, the degrees should be 0 (no rotation)
		if (!fromCamera && manufacturer.equalsIgnoreCase(SAMSUNG)) {
			degrees = 0;
		}

		return degrees;
	}

    /**
     * creates image on canvas the size of the specific screen size.
     *
     * @param activity       the activity of caller.
     * @param windowManager used to get size of screen.
     * @param myImage       the image from resources used as the background image.
     * @return Drawable which will be set as backgroundDrawable.
     */
    @SuppressLint("ResourceType")
    public static Drawable createImageOnCanvas(Activity activity, WindowManager windowManager, Bitmap myImage) {
        //getting the height of the actionBar (toolbar) to calculate the size of the picture.
        int actionBarHeight;
        final TypedArray styledAttributes =  AndroidFrameworkUtils.obtainStyledAttributesByResID(activity, android.R.attr.actionBarSize);
        actionBarHeight = (int) styledAttributes.getDimension(0, 0);
        styledAttributes.recycle();

        // getting screen size. height and width.
        int height = AndroidFrameworkUtils.getDisplayMetricsHeight(windowManager);
        int width = AndroidFrameworkUtils.getDisplayMetricsWidth(windowManager);

		//resizing the bitmap image to the appropriate size.
		Bitmap resizedBitmap = Bitmap.createScaledBitmap(
				myImage, width, height - actionBarHeight, false);

		//creating bitmap that will hold the canvas.
		Bitmap result = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
		Canvas canvas = new Canvas(result);

		//drawing the image at the starting point under the actionBar height.
		canvas.drawBitmap(resizedBitmap, width-resizedBitmap.getWidth(), height - resizedBitmap.getHeight(), null);

        //converting bitmap into drawable and return.
        return new BitmapDrawable(AndroidFrameworkUtils.getResources(), result);
    }

	/**
	 * Stack BlurUtil Algorithm by Mario Klingemann <mario@quasimondo.com>
	 * Ref: https://github.com/patrickfav/BlurTestAndroid/blob/master/BlurBenchmark/src/main/java/at/favre/app/blurbenchmark/blur/algorithms/StackBlur.java
	 * @param radius Decides how much blurry we want our Bitmap to be.
	 *               Higher the number, higher the blur effect.
	 * @param originalBitmap Bitmap to which apply the blur effect.
	 * @return Blurred Bitmap
	 */
	public static Bitmap createBlurredBitmap(int radius, Bitmap originalBitmap) {
		if (radius < 0) {
			radius = 0;
		}

		int w = originalBitmap.getWidth();
		int h = originalBitmap.getHeight();

		int[] pix = new int[w * h];
		originalBitmap.getPixels(pix, 0, w, 0, 0, w, h);

		int wm = w - 1;
		int hm = h - 1;
		int wh = w * h;
		int div = radius + radius + 1;

		int[] r = new int[wh];
		int[] g = new int[wh];
		int[] b = new int[wh];
		int rsum, gsum, bsum, x, y, i, p, yp, yi, yw;
		int[] v_min = new int[Math.max(w, h)];

		int divsum = (div + 1) >> 1;
		divsum *= divsum;
		int[] dv = new int[256 * divsum];
		for (i = 0; i < 256 * divsum; i++) {
			dv[i] = (i / divsum);
		}

		yw = yi = 0;

		int[][] stack = new int[div][3];
		int stackpointer;
		int stackstart;
		int[] sir;
		int rbs;
		int r1 = radius + 1;
		int routsum, goutsum, boutsum;
		int rinsum, ginsum, binsum;

		for (y = 0; y < h; y++) {
			rinsum = ginsum = binsum = routsum = goutsum = boutsum = rsum = gsum = bsum = 0;
			for (i = -radius; i <= radius; i++) {
				p = pix[yi + Math.min(wm, Math.max(i, 0))];
				sir = stack[i + radius];
				sir[0] = (p & 0xff0000) >> 16;
				sir[1] = (p & 0x00ff00) >> 8;
				sir[2] = (p & 0x0000ff);
				rbs = r1 - Math.abs(i);
				rsum += sir[0] * rbs;
				gsum += sir[1] * rbs;
				bsum += sir[2] * rbs;
				if (i > 0) {
					rinsum += sir[0];
					ginsum += sir[1];
					binsum += sir[2];
				} else {
					routsum += sir[0];
					goutsum += sir[1];
					boutsum += sir[2];
				}
			}
			stackpointer = radius;

			for (x = 0; x < w; x++) {

				r[yi] = dv[rsum];
				g[yi] = dv[gsum];
				b[yi] = dv[bsum];

				rsum -= routsum;
				gsum -= goutsum;
				bsum -= boutsum;

				stackstart = stackpointer - radius + div;
				sir = stack[stackstart % div];

				routsum -= sir[0];
				goutsum -= sir[1];
				boutsum -= sir[2];

				if (y == 0) {
					v_min[x] = Math.min(x + radius + 1, wm);
				}
				p = pix[yw + v_min[x]];

				sir[0] = (p & 0xff0000) >> 16;
				sir[1] = (p & 0x00ff00) >> 8;
				sir[2] = (p & 0x0000ff);

				rinsum += sir[0];
				ginsum += sir[1];
				binsum += sir[2];

				rsum += rinsum;
				gsum += ginsum;
				bsum += binsum;

				stackpointer = (stackpointer + 1) % div;
				sir = stack[(stackpointer) % div];

				routsum += sir[0];
				goutsum += sir[1];
				boutsum += sir[2];

				rinsum -= sir[0];
				ginsum -= sir[1];
				binsum -= sir[2];

				yi++;
			}
			yw += w;
		}
		for (x = 0; x < w; x++) {
			rinsum = ginsum = binsum = routsum = goutsum = boutsum = rsum = gsum = bsum = 0;
			yp = -radius * w;
			for (i = -radius; i <= radius; i++) {
				yi = Math.max(0, yp) + x;

				sir = stack[i + radius];

				sir[0] = r[yi];
				sir[1] = g[yi];
				sir[2] = b[yi];

				rbs = r1 - Math.abs(i);

				rsum += r[yi] * rbs;
				gsum += g[yi] * rbs;
				bsum += b[yi] * rbs;

				if (i > 0) {
					rinsum += sir[0];
					ginsum += sir[1];
					binsum += sir[2];
				} else {
					routsum += sir[0];
					goutsum += sir[1];
					boutsum += sir[2];
				}

				if (i < hm) {
					yp += w;
				}
			}
			yi = x;
			stackpointer = radius;
			for (y = 0; y < h; y++) {
				// Preserve alpha channel: ( 0xff000000 & pix[yi] )
				pix[yi] = (0xff000000 & pix[yi]) | (dv[rsum] << 16) | (dv[gsum] << 8) | dv[bsum];

				rsum -= routsum;
				gsum -= goutsum;
				bsum -= boutsum;

				stackstart = stackpointer - radius + div;
				sir = stack[stackstart % div];

				routsum -= sir[0];
				goutsum -= sir[1];
				boutsum -= sir[2];

				if (x == 0) {
					v_min[y] = Math.min(y + r1, hm) * w;
				}
				p = x + v_min[y];

				sir[0] = r[p];
				sir[1] = g[p];
				sir[2] = b[p];

				rinsum += sir[0];
				ginsum += sir[1];
				binsum += sir[2];

				rsum += rinsum;
				gsum += ginsum;
				bsum += binsum;

				stackpointer = (stackpointer + 1) % div;
				sir = stack[stackpointer];

				routsum += sir[0];
				goutsum += sir[1];
				boutsum += sir[2];

				rinsum -= sir[0];
				ginsum -= sir[1];
				binsum -= sir[2];

				yi += w;
			}
		}

		Bitmap bitmapCopy = originalBitmap.copy(originalBitmap.getConfig(), true);
		bitmapCopy.setPixels(pix, 0, w, 0, 0, w, h);
		return (bitmapCopy);
	}

	public static void clearImageRequests(Context context) {
		PicassoUtils.get(context).cancelTag(LP_PICASSO_TAG);
	}

}
