package com.liveperson.infra.utils;

import android.content.Context;
import androidx.annotation.NonNull;
import android.text.TextUtils;

import com.liveperson.infra.Infra;
import com.liveperson.infra.R;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPLog;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * This class holds various validations on a message.
 * Currently it has a maskMessage method that will mask any CC number in a message
 *
 * Created by nirni on 2/22/16.
 */
public class MessageValidator {

    private static final String TAG = "MessageValidator";
	private boolean mIsClientOnlyMaskingEnabled;
	private Pattern mClientOnlyRegexPattern = null;
	//the character to use for masking
	private final String mClientOnlyMaskCharacter;
	private final boolean mIsRealTimeMaskingEnabled;
	private Pattern mRealTimeRegexPattern = null;

	//the character to use for masking
	private final String mRealTimeMaskCharacter;

	private final String mClientOnlySystemMessage;
	private final String mRealTimeSystemMessage;

	public MessageValidator(boolean isAuthenticated, String realTimeSystemMessage, String clientOnlySystemMessage) {
		// Get message masking information
		mIsClientOnlyMaskingEnabled = Configuration.getBoolean(R.bool.enable_client_only_masking);
		mIsRealTimeMaskingEnabled = Configuration.getBoolean(R.bool.enable_real_time_masking);

		if (isAuthenticated){
			// When in authenticated mode, client-only masking is OFF regardless of the config file,
			// this is to prevent misuse of the feature and send unmasked data to the server that may be retrieved
			// by clients upon reinstall or login from another device, add this restriction to the feature documentation.
			mIsClientOnlyMaskingEnabled = false;
		}

		//the regex pattern to use for detecting what to mask
		String mClientOnlyMaskRegex = Configuration.getString(R.string.client_only_masking_regex);
		if (!TextUtils.isEmpty(mClientOnlyMaskRegex)){
			try{
				mClientOnlyRegexPattern = Pattern.compile(mClientOnlyMaskRegex);
			}catch (PatternSyntaxException exception){
				LPLog.INSTANCE.w(TAG, "Client only masking regex is invalid. aborting.", exception);
				mClientOnlyRegexPattern = null;
			}

		}

		//the regex pattern to use for detecting what to mask
		String mRealTimeMaskRegex = Configuration.getString(R.string.real_time_masking_regex);
		if (!TextUtils.isEmpty(mRealTimeMaskRegex)){
			try{
				mRealTimeRegexPattern = Pattern.compile(mRealTimeMaskRegex);
			}catch (PatternSyntaxException exception){
				LPLog.INSTANCE.w(TAG, "Real time masking regex is invalid. aborting.", exception);
				mRealTimeRegexPattern = null;
			}
		}

		Context context = Infra.instance.getApplicationContext();
		mClientOnlyMaskCharacter = context.getString(R.string.client_only_mask_character);
		mRealTimeMaskCharacter = context.getString(R.string.real_time_mask_character);

		mRealTimeSystemMessage = realTimeSystemMessage;
		mClientOnlySystemMessage = clientOnlySystemMessage;
	}

	/**
	 * Check If message to be sent is maskable (e.g: Not a SecureForm). If so, return masked message
	 * @param message
	 * @param isMaskable
	 * @return the masked message if it passes regex check or the same message
	 */

	public MaskedMessage maskMessage(String message, boolean isMaskable) {
		if ((mIsClientOnlyMaskingEnabled || mIsRealTimeMaskingEnabled) && isMaskable) {

			MaskedMessage maskedMessage = maskRealTimeMessage(message);
			if (maskedMessage == null) {
				maskedMessage = maskClientOnlyMessage(message);
			}

			if (maskedMessage != null) {
				return maskedMessage;
			}

		}
		// Credit card was not found in message. Return same received message
		return getDefaultNotMaskedMessage(message);
	}

	/**
	 * Mask a credit card number in a message if the message contains one.
	 * This method will return the message with masked CC number.
	 * If the message does not contain CC number it will return the same message received.
	 * @param message the message to check
	 * @return the masked message if it contains CC number or the received message if it does not contain one
	 */
	private MaskedMessage maskClientOnlyMessage(String message) {

		if (!mIsClientOnlyMaskingEnabled || mClientOnlyRegexPattern == null || (mClientOnlyMaskCharacter == null)) {
			return null;
		}

		Matcher matcher = mClientOnlyRegexPattern.matcher(message);

		if (matcher.find()) {

			// If found we construct a **** string with the same number of characters as the detected CC
			int startIndex = matcher.start();
			int endIndex = matcher.end();
			StringBuilder sb = new StringBuilder();
			for (int i = startIndex; i < endIndex; i++) {
				sb.append(mClientOnlyMaskCharacter);
			}

			// Replace CC number with '*'
			String maskedMessage = matcher.replaceAll(sb.toString());

			// In Client only masking we send the original message but store the masked message.
			return new MaskedMessage(maskedMessage, message, true, mClientOnlySystemMessage);

		}
		// Credit card was not found in message. Return same received message
		return null;
	}

	/**
	 * Mask a credit card number in a message if the message contains one.
	 * This method will return the message with masked CC number.
	 * If the message does not contain CC number it will return the same message received.
	 * @param message the message to check
	 * @return the masked message if it contains CC number or the received message if it does not contain one
	 */
	private MaskedMessage maskRealTimeMessage(String message) {

		if (!mIsRealTimeMaskingEnabled || mRealTimeRegexPattern == null || (mRealTimeMaskCharacter == null)) {
			return null;
		}

		Matcher matcher = mRealTimeRegexPattern.matcher(message);

		if (matcher.find()) {

			// If found we construct a **** string with the same number of characters as the detected CC
			int startIndex = matcher.start();
			int endIndex = matcher.end();
			StringBuilder sb = new StringBuilder();
			for (int i = startIndex; i < endIndex; i++) {
				sb.append(mRealTimeMaskCharacter);
			}

			// Replace CC number with '*'
			String maskedMessage = matcher.replaceAll(sb.toString());

			// In RealTime masking we won't send or store the original message at all. only the masked message.
			return new MaskedMessage(maskedMessage, maskedMessage, true, mRealTimeSystemMessage);

		}
		// Credit card was not found in message. Return same received message
		return null;
	}

	@NonNull
	private MaskedMessage getDefaultNotMaskedMessage(String message) {
		return new MaskedMessage(message, message, false, null);
	}


}
