package com.liveperson.infra.utils

import android.util.Base64
import com.liveperson.infra.model.PKCEParams
import com.nimbusds.jose.EncryptionMethod
import com.nimbusds.jose.JWEAlgorithm
import com.nimbusds.jose.JWEHeader
import com.nimbusds.jose.JWEObject
import com.nimbusds.jose.Payload
import com.nimbusds.jose.crypto.RSAEncrypter
import com.nimbusds.jose.jwk.JWK
import com.nimbusds.jose.jwk.RSAKey
import java.io.UnsupportedEncodingException
import java.security.MessageDigest
import java.security.NoSuchAlgorithmException
import java.security.SecureRandom

object PKCEUtils {
    private val JWE_ALGORITHM = JWEAlgorithm.RSA_OAEP_256
    private val JWE_ENCRYPTION_METHOD = EncryptionMethod.A256GCM


    /**
     * Creates a code challenge derived from the codeVerifier
     * code_challenge = BASE64URL-ENCODE(SHA256(ASCII(code_verifier)))
     */
    @Throws(UnsupportedEncodingException::class, NoSuchAlgorithmException::class)
    private fun generateCodeChallenge(codeVerifier: String): String {
        val bytes = codeVerifier.toByteArray(charset("US-ASCII"))
        val messageDigest: MessageDigest = MessageDigest.getInstance("SHA-256")
        messageDigest.update(bytes, 0, bytes.size)
        val digest: ByteArray = messageDigest.digest()
        val base64Flags = Base64.NO_PADDING or Base64.NO_WRAP or Base64.URL_SAFE
        return Base64.encodeToString(digest, base64Flags)
    }

    /**
     * Creates a high-entropy cryptographic random STRING using the
     * unreserved characters [A-Z] / [a-z] / [0-9] / "-" / "." / "_" / "~"
     * from Section 2.3 of [RFC3986], with a minimum length of 43 characters
     * and a maximum length of 128 characters.
     */
    @Throws(UnsupportedEncodingException::class)
    private fun generateCodeVerifier(): String {
        val minLength = 43
        val maxLength = 128

        val secureRandom = SecureRandom()
        val codeVerifierLength = secureRandom.nextInt(maxLength - minLength + 1) + minLength
        val codeVerifier = ByteArray(128)
        secureRandom.nextBytes(codeVerifier)
        val base64Flags = Base64.NO_PADDING or Base64.NO_WRAP or Base64.URL_SAFE
        // Base64 encodes every 3 bytes of input data into 4 characters of Base64 encoding
        // Each Base64 character represents 6 bits of data.
        return Base64.encodeToString(codeVerifier, base64Flags).substring(0, codeVerifierLength)
    }

    /**
     * Creates a custom code verifier in JWE format
     */
    private fun generateEncryptedCodeVerifier(codeVerifier: String, kId: String, jwk: JWK): String {
        // defined json format by LP IDP service
        val jsonCodeVerifier = "{\"code_verifier\":\"$codeVerifier\"}"

        val header = JWEHeader.Builder(JWE_ALGORITHM, JWE_ENCRYPTION_METHOD)
            .keyID(kId)
            .build()
        val payload = Payload(jsonCodeVerifier)
        val jweObject = JWEObject(header, payload)

        return if (jwk is RSAKey) {
            val encrypter = RSAEncrypter(jwk.toRSAPublicKey())
            jweObject.encrypt(encrypter)

            jweObject.serialize()
        } else {
            ""
        }
    }

    /**
     * Creates code challenge and custom code verifier to return to host app
     */
    fun generatePKCEParams(kId: String, jwk: JWK): PKCEParams {
        val codeVerifier = generateCodeVerifier()
        val codeChallenge = generateCodeChallenge(codeVerifier)
        val strJWE = generateEncryptedCodeVerifier(codeVerifier, kId, jwk)

        return PKCEParams(strJWE, codeChallenge)
    }
}