package com.liveperson.infra.database;

import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.LPLog;

import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;

/**
 * Created by Ilya Gazman on 9/9/2015.
 *
 * This class handling the execution of database tasks
 */
public class DataBaseExecutor {

    private static final String TAG = "DataBaseExecutor";

    private static final ThreadFactory threadFactory = r -> {
        SecurityManager s = System.getSecurityManager();
        ThreadGroup group = (s != null) ? s.getThreadGroup() : Thread.currentThread().getThreadGroup();
        Thread thread = new Thread(group, r, TAG, 0);
        if (thread.isDaemon())
            thread.setDaemon(false);
        if (thread.getPriority() != Thread.NORM_PRIORITY)
            thread.setPriority(Thread.NORM_PRIORITY);
        return thread;
    };

    private static ScheduledExecutorService service = Executors.newSingleThreadScheduledExecutor(threadFactory);

    private static void recreateServiceIfTerminated() {
        if (service.isTerminated() || service.isShutdown()) synchronized (DataBaseExecutor.class) {
            service = Executors.newSingleThreadScheduledExecutor(threadFactory);
        }
    }

    /**
     * Will execute on database thread or will run on the current thread if you already on database thread
     */
    public static void execute(final Runnable runnable) {
        try {
            recreateServiceIfTerminated();
            Runnable newRunnable = () -> {
                if (Thread.interrupted()) {
                    return;
                }
                runnable.run();
            };
            if (Thread.currentThread().getName().equals(TAG)) {
                newRunnable.run();
            } else {
                service.execute(runnable);
            }
        } catch (RejectedExecutionException exception) {
            LPLog.INSTANCE.d(TAG, "Failed to run database command", exception);
        }
    }

    public static void killAll(final ICallback<Void, Exception> callback) {
        try {
            if (service.isShutdown() || service.isTerminated()) {
                callback.onSuccess(null);
            } else synchronized (DataBaseExecutor.class) {
                service.shutdownNow();
            }
        } catch (Exception exception) {
            callback.onError(exception);
        }
    }
}
