package com.liveperson.infra.controller

import android.annotation.TargetApi
import android.content.res.Resources
import android.os.Build
import android.security.KeyPairGeneratorSpec
import android.security.keystore.KeyGenParameterSpec
import android.security.keystore.KeyPermanentlyInvalidatedException
import android.security.keystore.KeyProperties
import android.text.TextUtils
import android.util.Base64
import com.liveperson.infra.Clearable
import com.liveperson.infra.Infra
import com.liveperson.infra.R
import com.liveperson.infra.configuration.Configuration
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.FlowTags
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.log.LPLog.e
import com.liveperson.infra.log.LPLog.i
import com.liveperson.infra.log.LPLog.mask
import com.liveperson.infra.log.LPLog.v
import com.liveperson.infra.log.LPLog.w
import com.liveperson.infra.managers.PreferenceManager
import com.liveperson.infra.utils.AndroidFrameworkUtils
import com.liveperson.infra.utils.EncryptionVersion
import com.liveperson.infra.utils.Utils
import java.io.ByteArrayInputStream
import java.io.ByteArrayOutputStream
import java.io.IOException
import java.math.BigInteger
import java.nio.charset.StandardCharsets
import java.security.*
import java.security.spec.AlgorithmParameterSpec
import java.security.spec.MGF1ParameterSpec
import java.util.*
import javax.crypto.*
import javax.crypto.spec.GCMParameterSpec
import javax.crypto.spec.IvParameterSpec
import javax.crypto.spec.OAEPParameterSpec
import javax.crypto.spec.PSource
import javax.crypto.spec.SecretKeySpec
import javax.security.auth.x500.X500Principal

/**
 * LivePerson Encryption Service.
 *
 *
 * Creating an instance of this class will load (or create) an encryption key and use it to encrypt
 * and decrypt messages as they are passed through this class.
 */
open class DBEncryptionService : Clearable {

	companion object {

		private const val TAG = "DBEncryptionService"
		private const val ANDROID_KEYSTORE = "AndroidKeyStore"
		private const val DB_ENCRYPTION_KEY = "dbEncryptionKey"
		private const val DB_ENCRYPTION_USES_KEYSTORE = "dbEncryptionUsesKeyStore"
		private const val ANDROID_INFRA_DB_ENC_KEY = "androidInfraDbEncKey"
		private const val DB_ENC_CN = "CN=DBKeyEncryptor, O=Liveperson"
		private const val MDNAME = "SHA-256"
		private const val MGFNAME = "MGF1"
		private const val MGFSPEC_MDNAME = "SHA-1"
		private const val ANDROID_INFRA_DB_ENC_KEY_SIZE = 256

		/**
		 * This algorithm support API Levels 18+ (Android 4.3 and higher)
		 */
		private const val TRANSFORMATION_RSA_18_PLUS = "RSA/ECB/PKCS1Padding"

		/**
		 * This algorithm support API Levels 23+ (Android 6 and higher)
		 */
		private const val TRANSFORMATION_RSA_23_PLUS = "RSA/ECB/OAEPWithSHA-256AndMGF1Padding"
        /**
         * This algorithm support API Levels 10+ https://developer.android.com/reference/javax/crypto/Cipher
         */
		private const val TRANSFORMATION_AES_GCM_NoPadding="AES/GCM/NoPadding"

		private const val TRANSFORMATION_AES_PKCS5Padding = "AES/CBC/PKCS5Padding"
		private const val TRANSFORMATION_AES_PKCS7Padding = "AES/CBC/PKCS7Padding"

		private const val INITIALIZATION_VECTOR = "initializationVector"

		private const val IV_SEPARATOR = "::"

		private var encryptAttempts = 0

		/**
		 * Returns App Encryption Version 1. Currently not used to differentiate between versions.
		 *
		 * @return EncryptionVersion 1.
		 */
		@JvmStatic
		val appEncryptionVersion: EncryptionVersion
			get() = try {
				EncryptionVersion.fromInt(Configuration.getInteger(R.integer.encryptionVersion))
			} catch (e: Resources.NotFoundException) {
				e(TAG, ErrorCode.ERR_0000003C, "Exception while getting app encryption version.", e)
				EncryptionVersion.VERSION_1
			}
	}

	private lateinit var androidKeyStore: KeyStore

	private var dbEncryptionKey: SecretKey? = null

	private var legacyIvSpec: IvParameterSpec? = null

	private var onlyKeystore: Boolean = false

	private lateinit var cipherWrapperFactory: CipherWrapperFactory

	private lateinit var androidInterface: AndroidInterface

	/**
	 * If we get an encryption key from the host app, we don't save it locally.
	 * If we don't get it from the host app, we use our own generated key.
	 *
	 *
	 * On older devices (5.1.1/Lollipop or lower) where AndroidKeystore is supported, we save the
	 * generated key in SharedPreferences after encrypting the key with RSA KeyPair from the KeyStore.
	 * When they Keystore is not supported, the generated key is saved as is.
	 *
	 *
	 * On newer devices (6.0/Marshmallow and higher) we require use of the Keystore and always store
	 * our own generated keys inside the secure Keystore.
	 */
	init {
		initialize()
	}

	open fun initialize() {
		if (!this::androidInterface.isInitialized) {
			androidInterface = Android()
		}
		onlyKeystore =  // We can use the actually-secure system if and only if:
			Build.VERSION.SDK_INT >= Build.VERSION_CODES.M &&  // We're on a new-enough API
					!PreferenceManager.getInstance().contains( // And we don't have a legacy key
						DB_ENCRYPTION_KEY, PreferenceManager.APP_LEVEL_PREFERENCES
					)
		i(TAG, "Using " + (if (onlyKeystore) "Keystore" else "Legacy") + " encryption system.")

		androidInterface.loadKeystore()
		legacyIvSpec = loadLegacyIvSpec() // Hopefully this stays null, meaning we don't have one.

		if (!this::cipherWrapperFactory.isInitialized) {
			cipherWrapperFactory = CipherWrapperFactory()
		}

		if (onlyKeystore) {
			dbEncryptionKey = null // We no longer keep any keys in RAM.
		} else {
			setDbEncryptionKey(loadInternalLegacyAESKey())
		}
	}

	/**
	 * Loads an Initialization Vector, a 16-byte string of random junk used to kick off secure
	 * encryption under CBC mode. This junk must be newly re-randomized for each message encrypted.
	 * However, this method loads a single static IV used for all messages, which is super insecure.
	 *
	 *
	 * It is for this reason that this method has been reduced to only using a static IV if we
	 * already have one, and if we do not, then we don't load one. For all encryption operations
	 * that occur in the future, we must save a new Initialization Vector along with the encrypted
	 * message. For all encryption operations that occurred in the past, we must use this single
	 * insecure static IV.
	 *
	 *
	 * After calls to `logout()` or  on fresh launches of the SDK, this method will never load an IV.
	 */
	private fun loadLegacyIvSpec(): IvParameterSpec? {
		// Fill the initialization vector for the CBC algorithm the old/insecure way, if we have one
		val initializationVectorFromSharedPrefs = PreferenceManager.getInstance().getStringValue(
				INITIALIZATION_VECTOR, PreferenceManager.APP_LEVEL_PREFERENCES, null)

		if (initializationVectorFromSharedPrefs != null) {
			w(TAG, "Found a legacy Initialization Vector; loading it. " +
					"Please log out and back in to clear old data.")

			return IvParameterSpec(Base64.decode(initializationVectorFromSharedPrefs, Base64.DEFAULT))
		}

		return null
	}

	/**
	 * Converts a Base64-Encoded IVSpec String back into a usable IvParameterSpec object.
	 *
	 * @param base64ivSpec A cryptographic ivSpec, encoded in Base64.
	 * @return An IvParameterSpec containing the unpacked ivSpec.
	 */
	private fun unpackIvSpecBytes(base64ivSpec: String): IvParameterSpec {
		return IvParameterSpec(Base64.decode(base64ivSpec, Base64.DEFAULT))
	}

	/**
	 * Converts a Base64-Encoded IVSpec String back into a usable IvParameterSpec object.
	 *
	 * @param base64ivSpec A cryptographic ivSpec, encoded in Base64.
	 * @return An IvParameterSpec containing the unpacked ivSpec.
	 */
	private fun unpackGCMSpecBytes(base64ivSpec: String): GCMParameterSpec {
		return GCMParameterSpec(128, Base64.decode(base64ivSpec, Base64.DEFAULT))
	}

	/**
	 * Loads an AES encryption key to do cryptographic operations the old / insecure way, by keeping
	 * the key in memory, or creates one by calling `generateLegacyAesEncryptionKey()`. This key is
	 * returned as a Base64-encoded blob, and must be decoded before use. (use `setDbEncryptionKey`)
	 *
	 *
	 * To do things the new / secure way, use `createKeystoreAESKeyIfNecessary()` instead.
	 *
	 * @return A base-64 encoded copy of the legacy AES key.
	 */
	private fun loadInternalLegacyAESKey(): String? {
		// We didn't get the encryptionKey from the host app
		val encryptionKey: String?

		// Try to get the encrypted encryption key from shared prefs
		val encryptedEncryptionKeyFromSharedPrefs = PreferenceManager.getInstance().getStringValue(
				DB_ENCRYPTION_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, null)

		// Try to get the privateKey from the keyStore. This is done here only for testing whether
		// the keystore already holds the key.
		// In case of AutomaticRestore is enabled on the device, the restored data does not include
		// the keys from the keyStore, so if we don't have it, we behave as we need to recreate it.
		val privateKeyEntryForValidation: KeyStore.PrivateKeyEntry? = try {
			androidInterface.getKeystoreEntry()
		} catch (e: Exception) {
			null
		}

		// If either the key from
		if (encryptedEncryptionKeyFromSharedPrefs == null || privateKeyEntryForValidation == null) {
			// First time - doesn't exist in shared prefs yet
			encryptionKey = generateAesEncryptionKeyAndSave()

		} else {
			// Got the key from shared prefs
			if (PreferenceManager.getInstance().getBooleanValue(
							DB_ENCRYPTION_USES_KEYSTORE, PreferenceManager.APP_LEVEL_PREFERENCES, false)) {
				// If they key is encrypted, decrypt it.
				encryptionKey = decryptKey(encryptedEncryptionKeyFromSharedPrefs)

			} else { // The key is not encrypted. Use it as is.
				encryptionKey = encryptedEncryptionKeyFromSharedPrefs
				// The device now supports AndroidKeyStore. Probably after OS upgrade.
				// Encrypt & save the encrypted key instead of the unencrypted one.
				encryptAndSaveKey(encryptionKey)
			}
		}

		return encryptionKey
	}

	/**
	 *	Generate new KEY for message encryption/decryption
	 *	then encrypt the KEY using asymmetric keys
	 *	@see getCipherForKey
	 */
	private fun generateAesEncryptionKeyAndSave() : String {
		val encryptionKey = androidInterface.generateLegacyAesEncryptionKey()
		encryptAndSaveKey(encryptionKey)

		return encryptionKey
	}
	/**
	 * Init Cipher for key encryption/decryption
	 * Android API level < 23 (Android 6 - Marshmallow) will use @see TRANSFORMATION_RSA_18_PLUS
	 * Android API level >= 23 (Android 6 - Marshmallow) will use @see TRANSFORMATION_RSA_23_PLUS
	 */
	 private fun getCipherForKey(opMode: Int): CipherWrapper {
		val resultCipher: CipherWrapper
		var key: Key? = null

		// asymmetric uses public key to encrypt and private key to decrypt
		if (opMode == Cipher.ENCRYPT_MODE) {
			key = androidInterface.getPublicKey()
		}
		if (opMode == Cipher.DECRYPT_MODE) {
			key = androidInterface.getKeystoreEntry().privateKey
		}

		if (AndroidFrameworkUtils.sdkVersion >= Build.VERSION_CODES.M) {
			val spec = OAEPParameterSpec(
				MDNAME,
				MGFNAME,
				MGF1ParameterSpec(MGFSPEC_MDNAME),
				PSource.PSpecified.DEFAULT
			)
			resultCipher = cipherWrapperFactory.createCipherWrapper(
				TRANSFORMATION_RSA_23_PLUS,
				opMode,
				requireNotNull(key),
				spec
			)
		} else {
			resultCipher = cipherWrapperFactory.createCipherWrapper(
				TRANSFORMATION_RSA_18_PLUS,
				opMode,
				requireNotNull(key)
			)
		}
		return resultCipher
	}

	/**
	 * If we're using the insecure legacy encryption system, this method un-encrypts the AES key
	 * we stored in SharedPreferences with an RSA key stored in the Keystore
	 *
	 * @param encryptedKey The encrypted AES key material.
	 * @return Decrypted AES key material.
	 */
	private fun decryptKey(encryptedKey: String): String? {
		var decryptedKey: String? = null

		try {
			val output = getCipherForKey(Cipher.DECRYPT_MODE)
			decryptedKey = decryptKeyWithCipher(encryptedKey, output)
		} catch (e: IOException) {
			e(TAG, ErrorCode.ERR_00000035, "IOException while decrypting key. Android SDK Version: ${Build.VERSION.SDK_INT}", e)
			try {
				w(TAG,"fallback: use old RSA algorithm - RSA/ECB/PKCS1Padding, to decrypt key in sharedPref")
				val privateKeyEntry = androidInterface.getKeystoreEntry()

				val cipher = cipherWrapperFactory.createCipherWrapper(
					TRANSFORMATION_RSA_18_PLUS,
					Cipher.DECRYPT_MODE,
					privateKeyEntry.privateKey
				)
				decryptedKey = decryptKeyWithCipher(encryptedKey, cipher)

				// if Android SDK >= M, need to use keystore instead of sharedPref to store the key
				if (AndroidFrameworkUtils.sdkVersion >= Build.VERSION_CODES.M &&
					PreferenceManager.getInstance().contains(
						DB_ENCRYPTION_KEY, PreferenceManager.APP_LEVEL_PREFERENCES)) {
					d(TAG, "set flag to reset DBEncryptionService")
					PreferenceManager.getInstance().setBooleanValue(
						PreferenceManager.RESET_DB_ENCRYPTION_SERVICE_KEY,
						PreferenceManager.APP_LEVEL_PREFERENCES, true
					)
				}
				d(TAG, "Got decrypted key by using old RSA algorithm")
			} catch (e: Exception) {
				e(TAG, ErrorCode.ERR_00000035, "Exception while decrypting key - fallback", e)
			}
		} catch (e: Exception) {
			e(TAG, ErrorCode.ERR_00000035, "Exception while decrypting key.", e)
		}

		return decryptedKey
	}

	private fun decryptKeyWithCipher(encryptedKey: String, cipherWrapper: CipherWrapper): String {
		val cipherInputStream = cipherWrapper.createCipherInputStream(encryptedKey)
		val values = ArrayList<Byte>()
		var nextByte: Int

		while (cipherInputStream.read().also { nextByte = it } != -1) {
			values.add(nextByte.toByte())
		}
		val bytes = ByteArray(values.size)
		for (i in bytes.indices) {
			bytes[i] = values[i]
		}
		return String(bytes, 0, bytes.size, StandardCharsets.UTF_8)
	}

	/**
	 * Need to reset encryption service when device has Android SDK >= M but still using SharedPref to store key
	 * After reset, the device will move to use KeyStore to store key
	 */
	open fun resetDBEncryptionService() {
		d(TAG, "resetDBEncryptionService")
		onlyKeystore = true
		dbEncryptionKey = null
		legacyIvSpec = null
		androidInterface.clear()
		PreferenceManager.getInstance().remove(DB_ENCRYPTION_KEY, PreferenceManager.APP_LEVEL_PREFERENCES)
		PreferenceManager.getInstance().remove(INITIALIZATION_VECTOR, PreferenceManager.APP_LEVEL_PREFERENCES)
	}

	/**
	 * If we're using the insecure legacy encryption system, this method encrypts the exposed AES
	 * key material with a Keystore-backed RSA key, and stores the protected AES key in
	 * SharedPreferences.
	 *
	 * @param decryptedKey Exposed AES key material, to be encrypted and stored for later use.
	 */
	private fun encryptAndSaveKey(decryptedKey: String?) {
		var encryptedKey: String?
		var usesAndroidKeyStore = true

		generateKeyPairInStoreIfNotExists()

		try {
			val input = getCipherForKey(Cipher.ENCRYPT_MODE)
			val outputStream = ByteArrayOutputStream()
			val cipherOutputStream = input.createCipherOutputStream(outputStream)

			cipherOutputStream.write(decryptedKey!!.toByteArray(StandardCharsets.UTF_8))
			cipherOutputStream.close()

			encryptedKey = Base64.encodeToString(outputStream.toByteArray(), Base64.DEFAULT)
		} catch (e: Exception) {
			e(TAG, ErrorCode.ERR_00000036, "Exception while encrypting/saving key.", e)
			encryptedKey = decryptedKey
			usesAndroidKeyStore = false
		}

		// Save encrypted key:
		// put the encrypted key on shared prefs for the next time
		PreferenceManager.getInstance().setStringValue(
				DB_ENCRYPTION_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, encryptedKey)
		// Mark as encrypted using keystore
		PreferenceManager.getInstance().setBooleanValue(
				DB_ENCRYPTION_USES_KEYSTORE, PreferenceManager.APP_LEVEL_PREFERENCES, usesAndroidKeyStore)
	}



	/**
	 * This method verifies that the key pair exists in the Android key store.
	 * If not, it creates the pair.
	 */
	private fun generateKeyPairInStoreIfNotExists() {
		try {
			// Create new key if needed
			if (!androidInterface.keystoreContainsAlias(ANDROID_INFRA_DB_ENC_KEY)) {

				val start = Calendar.getInstance()
				val end = Calendar.getInstance()
				end.add(Calendar.YEAR, 120)

				val generator = androidInterface.initKeyPairGenerator(start, end)
				generator.generateKeyPair()
			}
		} catch (e: Exception) {
			e(TAG, ErrorCode.ERR_00000038, "Exception while generating KeyPair.", e)
		}
	}

	/**
	 * Unpacks a Base64-encoded AES key into exposed key material, and saves it as a local variable.
	 *
	 * @param key Base64-encoded AES material that will be decoded and held in a local variable.
	 */
	private fun setDbEncryptionKey(key: String?) {
		if (key.isNullOrEmpty()) {
			d(TAG, "setDbEncryptionKey - key is NullOrEmpty")
			return
		}

		val decodedKey = Base64.decode(key, Base64.DEFAULT)
		decodedKey[0]++
		dbEncryptionKey = SecretKeySpec(decodedKey, "AES")
	}

	/**
	 * Encrypts arbitrary data for storage at rest.
	 *
	 * @param plainText Unencrypted plain-text data that needs to be protected.
	 * @return A Base64-encoded-and-encrypted copy of the original plainText data.
	 */
	open fun encrypt(plainText: String?): String? {
		d(TAG, "encrypt()")
		if (TextUtils.isEmpty(plainText)) {
			return plainText
		}

		try {
			val cipher = getAESEncryptCipher()
			val encryptedBytes = cipher.doFinal(plainText!!.toByteArray(StandardCharsets.UTF_8))
			val iv = cipher.iv()

			val base64Bytes = Base64.encodeToString(encryptedBytes, Base64.DEFAULT)
			val base64iv = Base64.encodeToString(iv, Base64.DEFAULT)

			v(TAG, "Successfully Encrypted block " + mask(base64iv))
			encryptAttempts = 0
			return base64iv + IV_SEPARATOR + base64Bytes
		} catch (e: InvalidKeyException){
			// This exception appears on a launch after SDK version has been upgraded (5.17 -> 5.18). On Android OS 12+
			// In this case we need to delete key to let it be recreated on next attempt.
			// clear key and retry once.
			w(TAG, "InvalidKeyException $e")
			if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M && e.cause is KeyPermanentlyInvalidatedException) {
				e(TAG, ErrorCode.ERR_00000039, "KeyPermanentlyInvalidatedException while Encrypting text.", e)
				androidKeyStore.deleteEntry(ANDROID_INFRA_DB_ENC_KEY)
				if (encryptAttempts == 0) {
					encryptAttempts++
					encrypt(plainText)
				} else {
					return plainText
				}
			}
		} catch (e: Exception) {
			e(TAG, ErrorCode.ERR_00000039, "Exception while Encrypting text.", e)
		}

		return plainText
	}

	/**
	 * Un-encrypts arbitrary data for viewing by the user.
	 *
	 * @param cipherText Base64-encoded-and-encrypted data that needs to be read by a user.
	 * @return A plain-text copy of the original cipherText data.
	 */
	open fun decrypt(cipherText: String?): String? {
		d(TAG, "decrypt() " + cipherText)
		if (TextUtils.isEmpty(cipherText)) {
			return cipherText
		}

		val parts = cipherText!!.split(IV_SEPARATOR.toRegex()).toTypedArray()
		val gcmParameterSpec: GCMParameterSpec?
		val messageText: String

		if (parts.size > 1) { // We have both an ivSpec and an encrypted message
			gcmParameterSpec = unpackGCMSpecBytes(parts[0]);
			messageText = parts[1]
		} else {
			w(TAG, "decrypt failure. No gcmParameterSpec.")
			return cipherText
		}

		try {
			val cipherWrapper = getAESDecryptCipher(gcmParameterSpec)
			val data = Base64.decode(messageText, Base64.DEFAULT)
			val decryptedBytes = cipherWrapper.doFinal(data)
			val res = String(decryptedBytes, StandardCharsets.UTF_8)
			v(TAG, "Successfully Decrypted " + if (parts.size > 1) mask("block " + parts[0]+"\nres = " + res) else "Legacy block")
			return res
		} catch (e: InvalidAlgorithmParameterException){
			w(TAG, FlowTags.DECRYPTION, "InvalidAlgorithmParameterException "+cipherText)
			return null
		} catch (badPaddingException: BadPaddingException) {
			// Bug LE-91533 occurred
			w(TAG, FlowTags.DECRYPTION, "Caught a bad padding exception!", badPaddingException)
			d(TAG, FlowTags.DECRYPTION, "Using fallback after BadPaddingException")

			try {
				val cipher = getAESDecryptCipher(gcmParameterSpec)
				// Solution from: https://stackoverflow.com/questions/4580982/javax-crypto-badpaddingexception
				val data = Utils.hexStringToByteArray(messageText)
				val decryptedBytes = cipher.doFinal(data)

				d(TAG, FlowTags.DECRYPTION, "BadPaddingException fallback worked!")
				return String(decryptedBytes, StandardCharsets.UTF_8)

			} catch (e: Exception) {
				e(TAG, FlowTags.DECRYPTION, ErrorCode.ERR_0000003A, "BadPaddingException fallback failed.", e)
			}

		} catch (e: Exception) {
			e(TAG, FlowTags.DECRYPTION, ErrorCode.ERR_0000003B, "Caught an unexpected exception.", e)
		}

		return cipherText
	}

	override fun clear() {
		d(TAG, "clear()")
		androidInterface.clear()
	}

	/**
	 * Returns a Cipher prepared for Decryption operations, using whichever encryption method
	 * is currently possible given platform capabilities.
	 *
	 * @return A Cipher prepared with LP keys for encryption.
	 * @throws NoSuchAlgorithmException           If the device cannot use AES encryption.
	 * @throws NoSuchPaddingException             If the device cannot use PKCS5 or PKCS7 Padding
	 * @throws UnrecoverableEntryException        If the key cannot be read from the Keystore
	 * @throws KeyStoreException                  If the Keystore had a problem
	 * @throws InvalidAlgorithmParameterException If a param passed to the Cipher is invalid
	 * @throws InvalidKeyException                If the key used is invalid for this type of Cipher
	 */
	@Throws(
		NoSuchAlgorithmException::class,
		NoSuchPaddingException::class,
		UnrecoverableEntryException::class,
		KeyStoreException::class,
		InvalidAlgorithmParameterException::class,
		InvalidKeyException::class
	)
	private fun getAESEncryptCipher(): CipherWrapper {

		val cipherWrapper: CipherWrapper?

		if (!this::cipherWrapperFactory.isInitialized) {
			cipherWrapperFactory = CipherWrapperFactory()
		}
		if (onlyKeystore) {
			androidInterface.createKeystoreAESKeyIfNecessary()
			cipherWrapper = cipherWrapperFactory.createCipherWrapper(
				TRANSFORMATION_AES_GCM_NoPadding,
				Cipher.ENCRYPT_MODE,
				androidInterface.secretKey()
			)
		} else {
			cipherWrapper = cipherWrapperFactory.createCipherWrapper(
				TRANSFORMATION_AES_GCM_NoPadding,
				Cipher.ENCRYPT_MODE,
				dbEncryptionKey,
				legacyIvSpec
			)
		}
		return requireNotNull(cipherWrapper)
	}

	/**
	 * Returns a Cipher prepared for Decryption operations, using whichever encryption method
	 * is currently possible given platform capabilities.
	 *
	 * @param apSpec The Initialization Vector that was generated when the message you want this
	 * Cipher for was first encrypted.
	 *
	 * @return A Cipher prepared with LP keys for decryption.
	 *
	 * @throws NoSuchAlgorithmException           If the device cannot use AES encryption.
	 * @throws NoSuchPaddingException             If the device cannot use PKCS5 or PKCS7 Padding
	 * @throws UnrecoverableEntryException        If the key cannot be read from the Keystore
	 * @throws KeyStoreException                  If the Keystore had a problem
	 * @throws InvalidAlgorithmParameterException If a param passed to the Cipher is invalid
	 * @throws InvalidKeyException                If the key used is invalid for this type of Cipher
	 */
	@Throws(
		NoSuchAlgorithmException::class,
		NoSuchPaddingException::class,
		UnrecoverableEntryException::class,
		KeyStoreException::class,
		InvalidAlgorithmParameterException::class,
		InvalidKeyException::class
	)
	private fun getAESDecryptCipher(apSpec: AlgorithmParameterSpec?): CipherWrapper {
		val cipherWrapper: CipherWrapper

		if (onlyKeystore) {
			androidInterface.createKeystoreAESKeyIfNecessary()
			cipherWrapper = cipherWrapperFactory.createCipherWrapper(
				TRANSFORMATION_AES_GCM_NoPadding,
				Cipher.DECRYPT_MODE,
				androidInterface.secretKey(),
				requireNotNull(apSpec)
			)
		} else {
			cipherWrapper = cipherWrapperFactory.createCipherWrapper(
				TRANSFORMATION_AES_GCM_NoPadding,
				Cipher.DECRYPT_MODE,
				dbEncryptionKey,
				requireNotNull(apSpec)
			)
		}

		return cipherWrapper
	}

	open inner class Android : AndroidInterface {


		val keyStoreSecretKey: SecretKey
			get() {
				val keyEntry = androidKeyStore.getEntry(ANDROID_INFRA_DB_ENC_KEY, null)
				return (keyEntry as KeyStore.SecretKeyEntry).secretKey
			}

		private var secretKey: SecretKey? = null
			get() {
				return if (field != null) {
					field
				} else {
					keyStoreSecretKey
				}
			}


		/**
		 * Checks to see if we already have an AES key saved in the Keystore, and if not then create
		 * the key there, ready for future use.
		 */
		@TargetApi(Build.VERSION_CODES.M)
		override fun createKeystoreAESKeyIfNecessary() {
			try {
				if (!androidInterface.keystoreContainsAlias(ANDROID_INFRA_DB_ENC_KEY)) {
					try {
						val keyGenerator = KeyGenerator.getInstance(
							KeyProperties.KEY_ALGORITHM_AES,
							ANDROID_KEYSTORE)

						val keyGenParameterSpec = KeyGenParameterSpec.Builder(
							ANDROID_INFRA_DB_ENC_KEY,
							KeyProperties.PURPOSE_ENCRYPT or KeyProperties.PURPOSE_DECRYPT)
							.setKeySize(ANDROID_INFRA_DB_ENC_KEY_SIZE)
							.setBlockModes(KeyProperties.BLOCK_MODE_GCM)
							.setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_NONE)
							.build()

						keyGenerator.init(keyGenParameterSpec)
						keyGenerator.generateKey()
						androidInterface.refreshKey()
					} catch (e: Exception) {
						e(TAG, ErrorCode.ERR_0000014B,
							"Fatal exception while generating new AES key: ", e)
					}
				}
			} catch (e: KeyStoreException) {
				e(TAG, ErrorCode.ERR_0000014A, "Fatal exception while accessing keystore: ", e)
			}
		}

		/**
		 * Generates AES key material the legacy / insecure way. Please prefer using the method
		 * `createKeystoreAESKeyIfNecessary()` to create the AES key material within the Keystore itself.
		 *
		 * @return Exposed AES key material.
		 */
		override fun generateLegacyAesEncryptionKey(): String {
			val secretKey: SecretKey = try {
				val keyGen = KeyGenerator.getInstance("AES")

				keyGen.init(256)
				keyGen.generateKey()

			} catch (noSuchAlgException: NoSuchAlgorithmException) {
				e(
					TAG,
					ErrorCode.ERR_00000037,
					"Exception while generating AES Encryption Key",
					noSuchAlgException
				)

				// This is an extremely hacky fallback, and we should come up with a better alternative.
				val randomBytes = ByteArray(32)
				SecureRandom().nextBytes(randomBytes)

				return Base64.encodeToString(randomBytes, Base64.DEFAULT)
			}
			return Base64.encodeToString(secretKey.encoded, Base64.DEFAULT)
		}

		override fun keystoreContainsAlias(alias: String): Boolean {
			return androidKeyStore.containsAlias(alias)
		}

		override fun initKeyPairGenerator(start:Calendar, end: Calendar): KeyPairGenerator {
			val generator = KeyPairGenerator.getInstance("RSA", "AndroidKeyStore")
			val keyGenParameterSpec : KeyGenParameterSpec
			if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
				keyGenParameterSpec = KeyGenParameterSpec.Builder(ANDROID_INFRA_DB_ENC_KEY, KeyProperties.PURPOSE_ENCRYPT or KeyProperties.PURPOSE_DECRYPT)
					.setCertificateSubject(X500Principal(DB_ENC_CN))
					.setCertificateSerialNumber(BigInteger.ONE)
					.setKeyValidityStart(start.time)
					.setKeyValidityEnd(end.time)
					.build()
				generator.initialize(keyGenParameterSpec)
			} else {
				// should be removed after minSDK become 23+
				generator.initialize(
					KeyPairGeneratorSpec.Builder(Infra.instance.applicationContext)
						.setAlias(ANDROID_INFRA_DB_ENC_KEY)
						.setSubject(X500Principal(DB_ENC_CN))
						.setSerialNumber(BigInteger.ONE)
						.setStartDate(start.time)
						.setEndDate(end.time)
						.build())
			}
			return generator
		}

		@Throws(IOException::class)
		override fun getKeystoreEntry(): KeyStore.PrivateKeyEntry {
			return androidKeyStore.getEntry(
				ANDROID_INFRA_DB_ENC_KEY,
				null
			) as KeyStore.PrivateKeyEntry
		}

		/**
		 * Loads an instance of the system's hardware-backed Keystore connector.
		 */
		override fun loadKeystore() {
			try {
				androidKeyStore = KeyStore.getInstance("AndroidKeyStore")
				androidKeyStore.load(null, null)
			} catch (e: Exception) {
				e(TAG, ErrorCode.ERR_00000034, "Failed to load Keystore.", e)
			}
		}

		override fun secretKey(): SecretKey? {
			return secretKey
		}

		override fun refreshKey() {
			secretKey = keyStoreSecretKey
		}

		override fun getPublicKey(): PublicKey {
			return getKeystoreEntry().certificate.publicKey
		}


		/**
		 * Clears all Keystore entries. Ignores entries in SharedPreferences, expecting the SharedPrefs
		 * service to be able to clear itself with a similar Clearable method.
		 */
		override fun clear() {
			try {
				androidKeyStore.deleteEntry(ANDROID_INFRA_DB_ENC_KEY)
				d(TAG, "clear succeed")
			} catch (kse: Exception) {
				d(TAG, "exception deleting key store entry: ", kse)
			}
		}

	}
}

interface AndroidInterface {
	fun createKeystoreAESKeyIfNecessary()
	fun secretKey(): SecretKey?
	fun refreshKey()
	fun getPublicKey(): PublicKey
	fun loadKeystore()
	fun generateLegacyAesEncryptionKey(): String
	fun keystoreContainsAlias(alias: String): Boolean
	fun initKeyPairGenerator(start:Calendar, end: Calendar): KeyPairGenerator
	@Throws(IOException::class)
	fun getKeystoreEntry(): KeyStore.PrivateKeyEntry
	fun clear()
}

open class CipherWrapper(var transformation: String, var opmode: Int, var key: Key?) {

	var mParams: AlgorithmParameterSpec? = null

	constructor(
		transformation: String,
		opmode: Int,
		key: Key?,
		params: AlgorithmParameterSpec?
	) : this(transformation, opmode, key) {
		mParams = params
	}

	lateinit var cipher: Cipher

	open fun init(): CipherWrapper {
		if (!this::cipher.isInitialized) {
			cipher = Cipher.getInstance(transformation)
		}
		if (mParams != null) {
			cipher.init(opmode, key, mParams)
		} else {
			cipher.init(opmode, key)
		}
		return this
	}

	@Throws(BadPaddingException::class)
	open fun doFinal(data: ByteArray?): ByteArray {
		return cipher.doFinal(data)
	}

	open fun iv(): ByteArray {
		return cipher.iv
	}

	open fun createCipherInputStream(encryptedKey: String): CipherInputStream {
		return CipherInputStream(
			ByteArrayInputStream(Base64.decode(encryptedKey, Base64.DEFAULT)), cipher
		)
	}

	open fun createCipherOutputStream(byteArrayOutputStream: ByteArrayOutputStream): CipherOutputStream {
		return CipherOutputStream(byteArrayOutputStream, cipher)
	}
}

open class CipherWrapperFactory {

	open fun createCipherWrapper(
		transformation: String,
		opmode: Int,
		key: Key?,
		params: AlgorithmParameterSpec?
	): CipherWrapper {
		return CipherWrapper(transformation, opmode, key, params).init()
	}

	open fun createCipherWrapper(
		transformation: String,
		opmode: Int,
		key: Key?
	): CipherWrapper {
		return CipherWrapper(transformation, opmode, key).init()
	}
}
