package com.liveperson.infra.network.http.requests

import com.liveperson.infra.Command
import com.liveperson.infra.ICallback
import com.liveperson.infra.log.LPLog.i
import com.liveperson.infra.log.LPLog.w
import com.liveperson.infra.network.http.HttpHandler
import com.liveperson.infra.network.http.request.HttpGetRequest
import com.liveperson.infra.otel.LPTraceType
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject

/**
 * Request to fetch connector ids associated with an account.
 */
class ACCDNConnectorsRequest(private val requestDomain: String?, val brandId: String?, val certificates: List<String>?, val callback: ICallback<String, Exception>) : Command {

    private val TAG = "ACCDNConnectorsRequest"
    private val ACCDN_URL = "https://%s/api/account/%s/configuration/le-connectors/all-connectors"

    override fun execute() {
        if (requestDomain?.isNotEmpty() == true) {

            i(TAG, "Sending ACCDN connectors request...")
            // Construct the URL to AC-CDN. Domain key: acCdnDomain
            val connectorsUrl = String.format(ACCDN_URL, requestDomain, brandId)

            val httpGetRequest = HttpGetRequest(connectorsUrl,
                LPTraceType.ACCDN_CONNECTORS_REQ)
            httpGetRequest.certificatePinningKeys = certificates

            httpGetRequest.callback = object : ICallback<String, Exception> {
                override fun onSuccess(value: String?) {
                    i(TAG, "Successfully got connector information: $value")
                    // We can get 200 (success) error code with error response message !!
                    if (value == null || responseContainsError(value)) {
                        callback.onError(Exception(value))
                    } else {
                        callback.onSuccess(value)
                    }
                }

                override fun onError(exception: Exception?) {
                    w(TAG, "Failed to get connector information: $exception")
                    callback.onError(exception)
                }
            }
            HttpHandler.execute(httpGetRequest)
        }
    }

    /**
     * Check If ACCDN Connector's response is a valid response.
     */
    private fun responseContainsError(value: String?): Boolean {
        try {
            val response = JSONArray(value)
            // If we got non-empty JSON array, it is valid response.
            if (response.length() > 0) {
                return false
            }
        } catch (exception: JSONException) {
            try {
                val response = JSONObject(value)
                // If connector is not available for an account, server will return an error as
                // JSON object: {"error":"The requested resource could not be found but may be available again in the future."}
                if (response.has("error")) {
                    w(TAG, "Got error response: ${response.get("error")}")
                    return true
                }
            } catch (exception: Exception) {
                return true
            }
        }
        return true
    }
}