package com.liveperson.infra.network.http.requests

import com.liveperson.infra.Command
import com.liveperson.infra.ICallback
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.network.http.HttpHandler
import com.liveperson.infra.network.http.body.LPJSONArrayBody
import com.liveperson.infra.network.http.request.HttpPostRequest
import com.liveperson.infra.otel.LPTraceType
import org.json.JSONArray
import org.json.JSONObject
import java.lang.Exception

class EventManagerRequest(val domain: String?, val accountId: String, val token: String, val messages: List<JSONObject>?,
                          val certificates: List<String>?, val callback: ICallback<String, Exception>?): Command {

    companion object {
        private const val TAG = "EventManagerRequest"
        private const val EVENT_MANAGER_URL = "https://%s/api/account/%s/events"
    }

    override fun execute() {
        if (messages.isNullOrEmpty()) {
            LPLog.w(TAG, "Domain or Message is empty. Did not send log to event manager.")
            return
        }

        val jsonArray = JSONArray()
        for (message in messages) {
            LPLog.d(TAG, "sending event: $message")
            jsonArray.put(message)
        }

        // Build the Uri for our upload endpoint
        val requestUrl = String.format(EVENT_MANAGER_URL, domain, accountId)

        // Build a Post request with appropriate pinning keys
        val postRequest = HttpPostRequest(requestUrl,
            LPTraceType.HTTP_REQUEST)
        postRequest.addHeader("Authorization", "Bearer $token")
        postRequest.setBody(LPJSONArrayBody(jsonArray))
        postRequest.certificatePinningKeys = certificates
        postRequest.callback = object : ICallback<String?, Exception?> {
            override fun onSuccess(response: String?) {
                LPLog.i(TAG, "Successfully uploaded report to event manager.")
                callback?.onSuccess(response)
            }

            override fun onError(exception: Exception?) {
                LPLog.i(TAG, "Error uploading report to event manager.", exception)
                callback?.onError(exception)
            }
        }

        // Execute HTTP Post
        HttpHandler.execute(postRequest)
    }
}