package com.liveperson.infra.utils

import android.content.Context
import android.net.Uri
import android.os.ParcelFileDescriptor
import androidx.annotation.VisibleForTesting
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.log.LPLog.e
import org.jetbrains.annotations.TestOnly
import java.io.File
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.IOException

object FileUtils {
    private const val TAG = "FileUtils"
    private val FILE_DIRS = arrayOf("/images", "/voice", "/documents")

    /**
     * Write the contents of internal file to the external file created.
     * @param context application context
     * @param privateDirectory app's private directory from where you need to copy the file
     * @param publicDirectory device's public directory (downloads/drive) where file is being copied to
     * @return
     */
    @JvmStatic
    fun writeFileContent(
        context: Context,
        privateDirectory: File?,
        publicDirectory: Uri?
    ): Boolean {
        var inputStream: FileInputStream? = null
        var parcelFileDescriptor: ParcelFileDescriptor? = null
        var fileOutputStream: FileOutputStream? = null
        try {
            inputStream = FileInputStream(privateDirectory)
            parcelFileDescriptor =
                publicDirectory?.let { context.contentResolver.openFileDescriptor(it, "w") }
            fileOutputStream = FileOutputStream(parcelFileDescriptor?.fileDescriptor)
            val b = ByteArray(1024 * 5)
            var len: Int
            while (inputStream.read(b).also { len = it } > 0) {
                fileOutputStream.write(b, 0, len)
            }
        } catch (e: Exception) {
            e(TAG, ErrorCode.ERR_00000012, "Failed to copy file from privateDir to publicDir. ", e)
            return false
        } finally {
            try {
                inputStream?.close()
                fileOutputStream?.close()
                parcelFileDescriptor?.close()
            } catch (exception: IOException) {
                e(
                    TAG,
                    ErrorCode.ERR_00000013,
                    "Failed to close fileOutputStream/parcelFileDescriptor ",
                    exception
                )
            }
        }
        return true
    }

    /**
     * Create new directory inside app's private directory and file if does not exists
     * @param context application context
     * @param path location of file to be created
     * @param filename Name of file to be created
     */
    @JvmStatic
    fun getFilePath(context: Context, path: String, filename: String?): File? {
        // Define directory path and file path
        val privateDir = File(context.applicationContext.filesDir.toString() + path)
        val filePath = File(privateDir, filename)

        // Create directory for the current brandId
        if (!privateDir.exists()) {
            if (!privateDir.mkdirs()) {
                e(TAG, ErrorCode.ERR_00000014, "getFilePath: folder could not be created")
                return null
            }
        }
        return filePath
    }

    @JvmStatic
    fun deleteFilesSync(root: File) {
        val path = root.path
        try {
            FILE_DIRS.forEach { File(path + it).deleteRecursively() }
        } catch (ex: IOException) {
            FILE_DIRS.forEach { deleteRecursively(File(path + it)) }
        }
    }

    @JvmStatic
    fun deleteFileAsync(root: File) {
        ThreadPoolExecutor.execute {
            deleteFilesSync(root)
        }
    }

    @VisibleForTesting
    @TestOnly
    @JvmStatic
    fun containsFiles(context: Context): Boolean {
        val path = context.filesDir.path
        return FILE_DIRS.map {
            File(path + it).exists()
        }.fold(false) { prev, current ->
            prev or current
        }
    }

    @VisibleForTesting
    @TestOnly
    @JvmStatic
    fun createTestFiles(context: Context, quantity: Int) {
        val path = context.filesDir.path
        FILE_DIRS.forEach { dir ->
            val folder = File(path + dir).also { if (!it.exists()) it.mkdirs() }
            repeat(quantity) { createTestFile(folder) }
        }
    }

    @VisibleForTesting
    @TestOnly
    @JvmStatic
    fun createTestFile(folder: File): File {
        val name = "${System.currentTimeMillis()}.tmp"
        return File(folder, name).also { it.exists() or it.createNewFile() }
    }

    @JvmStatic
    private fun deleteRecursively(fileOrDirectory: File?) {
        if (fileOrDirectory == null) {
            e(TAG, ErrorCode.ERR_00000143, "file path is null")
            return
        }
        if (fileOrDirectory.isDirectory) {
            d(TAG, "deleteRecursive: deleting directory: " + fileOrDirectory.absolutePath)
            val listFiles = fileOrDirectory.listFiles()
            if (listFiles == null) {
                e(TAG, ErrorCode.ERR_00000144, "File list is null")
                return
            }
            for (child in fileOrDirectory.listFiles()) {
                d(TAG, "deleteRecursive: deleting file: " + child.absolutePath)
                deleteRecursively(child)
            }
        }
        fileOrDirectory.delete()
    }
}