package com.liveperson.infra.loggos

import android.net.Uri
import com.liveperson.infra.ICallback
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.network.http.HttpHandler
import com.liveperson.infra.network.http.body.LPJSONArrayBody
import com.liveperson.infra.network.http.request.HttpPostRequest
import com.liveperson.infra.otel.LPTraceType
import org.json.JSONArray
import org.json.JSONObject

/**
 * Uploads a set of JSONObject messages to Loggos
 */
open class LoggosUploader {

	companion object {
		private const val TAG = "LoggosUploader"

		private const val LOGGOS_URL = "https://%s/api/loggos/log/"
		private const val APP_NAME = "appName"
		private const val MESSAGING_SDK = "messaging_SDK_Android"
	}

	open fun sendBulk(domain: String, messages: List<JSONObject>, certificates: List<String>, callback: Callback?) {
		if (domain.isEmpty() || messages.isEmpty()) {
			LPLog.w(TAG, "Domain or Message is empty. Did not send log to Loggos.")
			return
		}

		// Build a json array from individual message objects
		val jsonArray = JSONArray()
		for (message in messages) {
			jsonArray.put(message)
		}

		// Build the Uri for our upload endpoint
		val fullUri = Uri.parse(String.format(LOGGOS_URL, domain)).buildUpon()
				.appendQueryParameter(APP_NAME, MESSAGING_SDK)
				.build()

		// Build a Post request with appropriate pinning keys
		val postRequest = HttpPostRequest(fullUri.toString(),
			LPTraceType.HTTP_REQUEST)
		postRequest.setBody(LPJSONArrayBody(jsonArray))
		postRequest.certificatePinningKeys = certificates
		postRequest.callback = object : ICallback<String?, Throwable?> {
			override fun onSuccess(response: String?) {
				LPLog.i(TAG, "Successfully uploaded report to Loggos.")
				callback?.onSuccess(response)
			}

			override fun onError(exception: Throwable?) {
				LPLog.w(TAG, "Error uploading report to Loggos.", exception)
				callback?.onError(messages, exception)
			}
		}

		// Execute HTTP Post
		HttpHandler.execute(postRequest)
	}

	interface Callback {
		fun onSuccess(logId: String?)
		fun onError(messages: List<JSONObject>?, exception: Throwable?)
	}
}
