package com.liveperson.infra.utils

import android.os.Build
import android.util.SparseArray
import com.liveperson.infra.log.LPLog.w

object DeviceUtils {

    private const val TAG = "DeviceUtils"

    /**
     * Generate user's device OS name using OS version code.
     */
    fun getOsName(): Any {
        try {
            val version = AndroidVersion.fromSdkVersionCode(Build.VERSION.SDK_INT)
            return version.logName
        } catch (exception: Exception) {
            w(TAG, "getOsName: Failed to obtain device OS name: ", exception)
        }
        return ""
    }

    /**
     * Return Android OS name from Static array of all available versions.
     */
    private enum class AndroidVersion(val verCode: Int, val logName: String) {
        UNSUPPORTED(-1, "Unsupported version"),
        BASE(1, "Base (1.0)"),
        BASE_1_1(2,"Base (1.1)"),
        CUPCAKE(3, "Cupcake (1.5)"),
        DONUT(4, "Donut (1.6)"),
        ECLAIR(5, "Eclair (2.0)"),
        ECLAIR_0_1(6, "Eclair (2.0.1)"),
        ECLAIR_MR1(7, "Eclair (2.1)"),
        FROYO(8, "Froyo (2.2)"),
        GINGERBREAD(9, "Gingerbread (2.3)"),
        GINGERBREAD_MR1(10, "Gingerbread (2.3.3)"),
        HONEYCOMB(11, "Honeycomb (3.0)"),
        HONEYCOMB_MR1(12, "Honeycomb (3.1)"),
        HONEYCOMB_MR2(13, "Honeycomb (3.2)"),
        ICE_CREAM_SANDWICH(14, "Ice Cream Sandwich (4.0)"),
        ICE_CREAM_SANDWICH_MR1(15, "Ice Cream Sandwich (4.0.3)"),
        JELLY_BEAN(16, "Jelly Bean (4.1)"),
        JELLY_BEAN_MR1(17, "Jelly Bean (4.2)"),
        JELLY_BEAN_MR2(18, "Jelly Bean (4.3)"),
        KITKAT(19, "Kitkat (4.4)"),
        KITKAT_WATCH(20, "Kitkat Watch (4.4W)"),
        LOLLIPOP(21, "Lollipop (5.0)"),
        LOLLIPOP_MR1(22, "Lollipop (5.1)"),
        MARSHMALLOW(23, "Marshmallow (6.0)"),
        NOUGAT(24, "Nougat (7.0)"),
        NOUGAT_MR1(25, "Nougat (7.1)"),
        OREO(26, "Oreo (8.0)"),
        OREO_MR1(27, "Oreo (8.1)"),
        PIE(28, "Pie (9.0)"),
        Q(29, "Q (10.0)"),
        R(30, "R (11.0)"),
        S(31, "S (12.0)"),
        Sv2(32, "Snow Cone v2 (12.1)"),
        T(33, "Tiramisu (13.0)"),
        UPSIDE_DOWN_CAKE(34, "Upside Down Cake (14)");

        companion object {
            private val verCodeMap = SparseArray<AndroidVersion>()

            fun fromSdkVersionCode(verCode: Int): AndroidVersion {
                return verCodeMap.get(verCode, UNSUPPORTED)
            }

            init {
                for (version in values()) {
                    verCodeMap.put(version.verCode, version)
                }
            }
        }
    }
}