/* Copyright (c) LivePerson, Inc. */
package com.liveperson.infra.utils.markdown

import android.text.TextUtils
import com.liveperson.infra.utils.LinkUtils

/**
 * Created by Yanda on 05/11/2020
 */
class MarkdownHyperlinkItem {

	companion object {
		private const val EMPTY_STRING = ""
		private const val BRACKET_CHARACTER_LENGTH = 1 //The length for a single character. Such as '(', ')', '[', ']'
		private const val MIDDLE_BRACKETS = "]("
		private const val HTTP_STRING = "http://"
		private const val HTTPS_STRING = "https://"
	}

	var start: Int = -1
	var end: Int = -1
	var title: String? = null
		private set
	var link: String? = null
		private set

	/**
	 * Call this method when find an [LinkUtils.TAG_CLOSE] in message. Set [title] and [link].
	 *
	 * @param message The message that consumer received.
	 */
	fun buildHyperlinkItem(message: String) {
		val string = message.substring(start, end)
		val titleStartIndex = LinkUtils.TAG_OPEN.length + BRACKET_CHARACTER_LENGTH
		val middleBracketsIndexList = findMiddleBracketsIndexes(string)
		val lastUrlCharacterIndex = string.length - LinkUtils.TAG_CLOSE.length - BRACKET_CHARACTER_LENGTH

		for (middleBracketsIndex in middleBracketsIndexList) {
			//This checks if it's a valid link, in case the title includes brackets.
			title = string.substring(titleStartIndex, middleBracketsIndex)
			link = string.substring(middleBracketsIndex + MIDDLE_BRACKETS.length, lastUrlCharacterIndex)
			if (link == null) {
				link = EMPTY_STRING //set current item invalid.
				break
			}
			if (link!!.startsWith(HTTP_STRING, true) || link!!.startsWith(HTTPS_STRING, true)) {
				break
			}
		}
	}

	/**
	 * Check if current item is valid. Prevents wrong link from showing if consumer receive a wrong markdown hyperlink format message.
	 */
	fun isValid(): Boolean {
		return start != -1 && end > start && !TextUtils.isEmpty(title) && !TextUtils.isEmpty(link)
	}

	/**
	 * Find all possible indexes of [MIDDLE_BRACKETS] in a string.
	 *
	 * @param string The markdown hyperlink message.
	 * @return A list of indexes of [MIDDLE_BRACKETS]
	 */
	private fun findMiddleBracketsIndexes(string: String): List<Int> {
		val indexList = ArrayList<Int>()
		var index = LinkUtils.TAG_OPEN.length + BRACKET_CHARACTER_LENGTH
		while (index <= string.length - LinkUtils.TAG_CLOSE.length - MIDDLE_BRACKETS.length) {
			if (MIDDLE_BRACKETS == string.substring(index, index + MIDDLE_BRACKETS.length)) {
				indexList.add(index)
				index += MIDDLE_BRACKETS.length
			} else {
				index++
			}
		}
		return indexList
	}
}
