package com.liveperson.infra.analytics

import android.content.Context
import android.os.Build
import com.liveperson.infra.BuildConfig
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.log.LPLog.w
import com.liveperson.infra.managers.PreferenceManager
import com.liveperson.infra.managers.PreferenceManager.APP_ID_PREFERENCE_KEY
import com.liveperson.infra.utils.AnalyticsUtils
import com.liveperson.infra.utils.DeviceUtils.getOsName
import com.liveperson.infra.utils.VersionUtils.getAppVersion
import java.util.Locale
import java.util.TimeZone
import java.util.UUID
import java.util.concurrent.LinkedBlockingQueue

open class AnalyticsService {
    private lateinit var appContext: Context
    private lateinit var brandId: String
    private var appId: String = ""
    private var userProperties = HashMap<String, Any>()
    private var userEvents = LinkedBlockingQueue<AnalyticsEvent>()
    private var isAnalyticsEnabled = true // Analytics is enabled by default

    private var sessionId: String = ""

    companion object {
        private const val TAG = "AnalyticsService"

        private const val BRAND_ID = "accountID"
        private const val APP_ID = "hostAppName"
        private const val APP_VERSION = "hostAppVersion"
        private const val SDK_VERSION = "sdkVersion"
        private const val DEVICE_API = "deviceApi" // Not indexed in Kibana
        private const val DEVICE_LANGUAGE = "language"
        private const val DEVICE_SCRIPT = "deviceScript" // Not indexed in Kibana
        private const val DEVICE_REGION = "region"
        private const val DEVICE_TIME_ZONE = "geoip.timezone"
        private const val DEVICE_MAKE = "deviceFamily"
        private const val DEVICE_MODEL = "deviceModel"
        private const val DEVICE_OS = "deviceOS" // // Not indexed in Kibana
        private const val DEVICE_OS_VERSION = "deviceOSVersion"
        private const val LOG_TIME = "event_time"
        private const val SESSION_ID = "sessionId"
    }

    /**
     * Initialize the analytics service to track all user events during the time of interaction
     * with SDK.
     */
    fun init(applicationContext: Context, brandId: String, appId: String) {
        this.appContext = applicationContext
        this.brandId = brandId
        this.appId = appId
        setUserProperties()
    }

    /**
     * Add user event, appended with brandId and current time into list of events.
     */
    open fun logUserEvent(analyticsEvent: AnalyticsEvent?) {
        try {
            // Do not log event if analytics is disabled.
            if (!isAnalyticsEnabled || analyticsEvent == null) {
                d(TAG, "logUserEvent: Analytics is disabled. Do not log event.")
                return
            }
            if (!this::brandId.isInitialized) {
                d(TAG, "logUserEvent: We don't have brandId yet. Use empty")
                brandId = ""
            }
            // Append brandId and current time as event property
            val eventProperties = arrayOf(EventProperty(LOG_TIME, AnalyticsUtils.getStartTime()),
                    EventProperty(BRAND_ID, brandId)) + analyticsEvent.getEventProperties()

            d(TAG, "logUserEvent: Adding user event ${analyticsEvent.eventName} into the list.");
            userEvents.add(AnalyticsEvent(analyticsEvent.eventName, *eventProperties))
            d(TAG, "logUserEvent: Total events cached: " + userEvents.size)
        } catch (exception: Exception) {
            w(TAG, "logUserEvent: Failed to log user event: ", exception)
        }
    }


    /**
     * Create HashMap of user properties which is being sent along with user events.
     */
    private fun setUserProperties() {
        try {
            userProperties.put(APP_ID, getBrandAppId())
            userProperties.put(APP_VERSION, getAppVersion(appContext))
            userProperties.put(SDK_VERSION, BuildConfig.VERSION_NAME)
            userProperties.put(BRAND_ID, brandId)
            userProperties.put(DEVICE_API, Build.VERSION.SDK_INT)
            userProperties.put(DEVICE_LANGUAGE, Locale.getDefault().toString())
//            userProperties.put(DEVICE_SCRIPT, getDeviceScript()) // Revisit this
            userProperties.put(DEVICE_REGION, Locale.getDefault().country)
            userProperties.put(DEVICE_TIME_ZONE, TimeZone.getDefault().id)
            userProperties.put(DEVICE_MAKE, Build.MANUFACTURER)
            userProperties.put(DEVICE_MODEL, Build.MODEL)
            userProperties.put(DEVICE_OS, getOsName())
            userProperties.put(DEVICE_OS_VERSION, Build.VERSION.RELEASE)
            userProperties.put(SESSION_ID, getSessionId())
        } catch (e: Exception) {
            w(TAG, "setUserProperties: Exception while mapping user properties.", e)
        }
    }

    /**
     * Set brand Id in case init is not yet called (mostly because SDK was in background)
     */
    fun setBrandId(brandId: String?) {
        if (brandId != null && brandId.isNotEmpty()) {
            this.brandId = brandId
        }
    }

    /**
     * Set host app context If init is not called yet (mostly because SDK was in background)
     */
    fun setAppContext(appContext: Context?) {
        if (appContext != null) {
            this.appContext = appContext
        }
    }

    /**
     * Set If analytics is enabled in site settings
     */
    fun setIsAnalyticsEnabled(isEnabled: Boolean) {
        isAnalyticsEnabled = isEnabled
    }

    /**
     * Return brand's appId from shared preference If missing locally
     */
    private fun getBrandAppId(): String {
        if (appId.isEmpty() && brandId.isNotEmpty()) {
            d(TAG, "getBrandAppId: Missing appId, trying to get it from shared preference.")
            appId = PreferenceManager.getInstance().getStringValue(APP_ID_PREFERENCE_KEY, brandId, "")
        }
        return appId
    }

    /**
     * Generate unique sessionId to track user events for each session
     */
    private fun getSessionId(): String {
        if (sessionId.isEmpty()) {
            sessionId = UUID.randomUUID().toString()
        }
        return sessionId
    }
}