@file:JvmName("HtmlUtils")

package com.liveperson.infra.html

import android.os.Build
import android.text.Html
import android.text.SpannableString
import android.text.SpannableStringBuilder
import com.liveperson.infra.html.tag.LI_TAG
import com.liveperson.infra.html.tag.ListTagHandler
import com.liveperson.infra.html.tag.OL_TAG
import com.liveperson.infra.html.tag.UL_TAG
import com.liveperson.infra.log.LPLog

private const val TAG = "HtmlUtils"

/**
 * Extension function used to convert raw string with html tags to spannable string.
 *
 * This method uses custom [tag handler][ListTagHandler] to represent ordered
 * and unordered lists properly and customize the behavior of their representation.
 *
 * @return spannable string without any html tags.
 */
fun String.convertHtmlToStyledString(): SpannableString {
    val formattedString = "<div>$this</div>"
        .replace("(?i)<ul[^>]*>".toRegex(), "<$UL_TAG>")
        .replace("(?i)</ul>".toRegex(), "</$UL_TAG>")
        .replace("(?i)<ol[^>]*>".toRegex(), "<$OL_TAG>")
        .replace("(?i)</ol>".toRegex(), "</$OL_TAG>")
        .replace("(?i)<li[^>]*>".toRegex(), "<$LI_TAG>")
        .replace("(?i)</li>".toRegex(), "</$LI_TAG>")

    val styledString: CharSequence = try {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            Html.fromHtml(formattedString, Html.FROM_HTML_MODE_COMPACT, null, ListTagHandler())
        } else {
            @Suppress("DEPRECATION")
            Html.fromHtml(formattedString, null, ListTagHandler())
        }
    } catch (throwable: Throwable) {
        LPLog.d(TAG, "Error occurred while converting raw html to styled string", throwable)
        this
    }
    LPLog.d(TAG, "Styled string $styledString")
    return SpannableString(styledString)
}