package com.liveperson.infra.network.http;


import androidx.annotation.NonNull;
import android.text.TextUtils;
import android.util.Pair;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.otel.LPTelemetryAttributeKey;
import com.liveperson.infra.otel.LPTelemetryManager;
import com.liveperson.infra.otel.LPTraceSpan;
import com.liveperson.infra.otel.models.OtlpAttribute;
import com.liveperson.infra.otel.models.OtlpValueData;
import com.liveperson.infra.network.http.client.OkHttpClientHolder;
import com.liveperson.infra.network.http.client.TimeoutInterceptor;
import com.liveperson.infra.network.http.body.HttpRequestBody;
import com.liveperson.infra.network.http.request.HttpRequest;
import com.liveperson.infra.utils.ThreadPoolExecutor;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Headers;
import okhttp3.MediaType;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;


/**
 * Created by ofira on 02/11/2015.
 */
public class HttpHandler {

    private static final String TAG = "HttpHandler";

    /**
     * Execute http request with delay
     *
     * @param request - {@link HttpRequest}
     * @param delay   - long value to define delay
     */
    public static void executeDelayed(final HttpRequest request, long delay) {
        Runnable httpRunnable = buildRunnableByType(request);
        ThreadPoolExecutor.executeDelayed(httpRunnable, delay);
    }

    /**
     * Execute http request immediately
     *
     * @param request - {@link HttpRequest}
     */
    public static void execute(final HttpRequest request) {
        executeDelayed(request, 0);
    }

    public static void cancelAllCalls() {
        OkHttpClientHolder.cancelAllRequests();
    }

    private static Runnable buildRunnableByType(final HttpRequest request) {
        return () -> {
            Request httpReq = HttpRequestBuilder.build(request);

            // Add all interceptors to clientBuilder
	        LPLog.INSTANCE.i(TAG, "URL: " + httpReq.url().host());

            //Certificate Pinning, checking if we have a valid list
	        LPLog.INSTANCE.d(TAG, "Sending http request: " + httpReq.url() +
                     ((request.getCertificatePinningKeys() != null && !request.getCertificatePinningKeys().isEmpty() ) ?  " with Pinning Keys " +  LPLog.INSTANCE.mask(TextUtils.join(",", request.getCertificatePinningKeys())) : " with no Pinning Keys"));

	        if (!InternetConnectionService.isNetworkAvailable()) {
                LPLog.INSTANCE.w(TAG, "No network connection.");
                request.processErrorResponse(new Exception("No network connection."));
                return;
            }

            List<OtlpAttribute> otlpAttributes = new ArrayList<>();
            otlpAttributes.add(new OtlpAttribute(LPTelemetryAttributeKey.URL_FULL.getValue(), new OtlpValueData.StringValue(request.getUrl())));
            otlpAttributes.add(new OtlpAttribute(LPTelemetryAttributeKey.METHOD.getValue(), new OtlpValueData.StringValue(request.getMethod().name())));
            LPTraceSpan traceSpan = LPTelemetryManager.INSTANCE.begin(request.getTraceType(), otlpAttributes);

            OkHttpClientHolder.proceed(httpReq, request.getCertificatePinningKeys(), new Callback() {
                @Override
                public void onFailure(@NonNull Call call, @NonNull IOException e) {
	                LPLog.INSTANCE.w(TAG, "onFailure ", e);
                    if (traceSpan != null) {
                        traceSpan.cancel();
                    }
                    request.processErrorResponse(e);
                }

                @Override
                public void onResponse(@NonNull Call call, @NonNull Response response) {
	                LPLog.INSTANCE.i(TAG, "onResponse from URL: " + response.request().url());
                    if (traceSpan != null) {
                        traceSpan.end();
                    }
                    request.processResponse(response);
                }
            });
        };
    }

    private static class HttpRequestBuilder {

        public static Request build(HttpRequest request) {
            Request.Builder httpBuilder = new Request.Builder();
            httpBuilder.url(request.getUrl());

            switch (request.getMethod()) {
                case GET:
                    httpBuilder.get();
                    break;
                case HEAD:
                    httpBuilder.head();
                    break;
                case POST:
                    httpBuilder.post(getRequestBody(request));
                    break;
                case PUT:
                    httpBuilder.put(getRequestBody(request));
                    break;
                case DELETE:
                    httpBuilder.delete();
                    break;
            }

            // Add headers
            if (!request.getHeaders().isEmpty()) {
                Headers.Builder headersBuilder = new Headers.Builder();
                for (Pair<String, String> header : request.getHeaders()) {
                    LPLog.INSTANCE.d(TAG, "header.first " + header.first + " header.second " + header.second);
                    if (header.first != null && header.second != null) try {
                        headersBuilder.add(header.first, header.second);
                    } catch (IllegalArgumentException illegalStateException) {
                        if (illegalStateException.getMessage() != null
                                && illegalStateException.getMessage().contains("Unexpected char")) {
                            headersBuilder.addUnsafeNonAscii(header.first, header.second);
                            LPLog.INSTANCE.w(TAG, illegalStateException.getMessage());
                        }
                    }
                }
                httpBuilder.headers(headersBuilder.build());
            }
            httpBuilder.addHeader(TimeoutInterceptor.HEADER_TIMEOUT, String.valueOf(request.getTimeout()));
            return httpBuilder.build();
        }

        @NonNull
        private static RequestBody getRequestBody(HttpRequest request) {
            MediaType type;
            RequestBody reqBody;
            HttpRequestBody requestBody = request.getRequestBody();
            if (requestBody != null) {
                type = MediaType.parse(requestBody.getContentType());
                if (requestBody.isString()) {
                    reqBody = RequestBody.create(type, (String) requestBody.get());
                } else {
                    reqBody = RequestBody.create(type, (byte[]) requestBody.get());
                }
            } else {
                reqBody = RequestBody.create(null, new byte[0]);
            }
            return reqBody;
        }
    }
}
