package com.liveperson.infra.network.http.client

import com.liveperson.infra.log.LPLog
import okhttp3.Callback
import okhttp3.CertificatePinner
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.WebSocket
import okhttp3.WebSocketListener
import java.util.concurrent.TimeUnit

object OkHttpClientHolder {

    private const val TAG = "OkHttpClientHolder"

    private const val PERIODIC_PING_TIME = 20 * 1000L
    // default connect timeout if it wasn't specified in request
    private const val CONNECT_TIMEOUT_IN_MINUTES = 1L
    // default read timeout if it wasn't specified in request
    private const val READ_TIMEOUT_IN_MINUTES = 1L
    // default write timeout if it wasn't specified in request
    private const val WRITE_TIMEOUT_IN_MINUTES = 1L

    private var _okHttpClient: OkHttpClient = buildNewClient()

    @Synchronized
    @JvmStatic
    fun setInterceptors(interceptors: List<Interceptor>) {
        val oldInterceptors = _okHttpClient.interceptors
        if (oldInterceptors.containsAll(interceptors)) {
            LPLog.d(TAG, "Interceptors are the same. Skipping building new client.")
            return
        }
        val builder = _okHttpClient.newBuilder()

        interceptors.filterNot { oldInterceptors.contains(it) }
            .forEach { builder.addInterceptor(it) }
        _okHttpClient = builder.build()
    }

    @Synchronized
    @JvmStatic
    fun proceed(request: Request, keyPairs: List<String>, callback: Callback) {
        val requestKeys = keyPairs.fold(CertificatePinner.Builder()) { cert, keyPair ->
            val pair = keyPair.split(";")
            if (pair.size == 2) {
                val (domain, key) = pair
                cert.add(domain, key)
            } else {
                cert
            }
        }
        val newPinner = requestKeys.build()
        if (_okHttpClient.certificatePinner.pins == newPinner.pins) {
            LPLog.d(TAG, "Pinning keys are the same. Skipping building new client.")
        } else {
            _okHttpClient = _okHttpClient.newBuilder()
                .certificatePinner(newPinner)
                .build()
        }
        _okHttpClient.newCall(request).enqueue(callback)
    }

    @Synchronized
    @JvmStatic
    fun cancelAllRequests() {
        _okHttpClient.dispatcher.cancelAll()
    }

    @Synchronized
    @JvmStatic
    fun newSocket(request: Request, listener: WebSocketListener): WebSocket {
        return _okHttpClient.newWebSocket(request, listener)
    }

    private fun buildNewClient(): OkHttpClient {
        val builder =  OkHttpClient.Builder()
        return builder.addInterceptor(TimeoutInterceptor())
            .pingInterval(PERIODIC_PING_TIME, TimeUnit.MILLISECONDS)
            .connectTimeout(CONNECT_TIMEOUT_IN_MINUTES, TimeUnit.MINUTES)
            .readTimeout(READ_TIMEOUT_IN_MINUTES, TimeUnit.MINUTES)
            .writeTimeout(WRITE_TIMEOUT_IN_MINUTES, TimeUnit.MINUTES)
            .followRedirects(true)
            .build()
    }
}