package com.liveperson.infra;

import android.os.Bundle;
import android.text.TextUtils;

import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;
import com.liveperson.infra.utils.LocalBroadcast;

import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Created by shiranr on 12/11/2015.
 * Modified by nirni
 * <p/>
 * This is used to register any view as the one in the foreground. The view that registered needs
 * to provide the brand ID which it uses.
 * <p/>
 * When a view is registered/unregistered a corresponding broadcast is sent.
 */
public class ForegroundService implements ShutDown, ForegroundServiceInterface {

    private static final String TAG = "ForegroundService";
    public static final String SCREEN_FOREGROUND_ACTION = "SCREEN_FOREGROUND_ACTION";
    public static final String SCREEN_FOREGROUND_EXTRA_BRAND_ID = "brand_id";
    public static final String SCREEN_FOREGROUND_EXTRA_TARGET_ID = "target_id";
    public static final String SCREEN_FOREGROUND_EXTRA_IS_FOREGROUND = "is_foreground";
    private HashSet<String> mForegroundBrands = new HashSet<>();
    private Map<String, Long> mForegroundMillis = new ConcurrentHashMap<>();
    private HashSet<String> mForegroundTargets = new HashSet<>();

    private static volatile ForegroundService Instance = null;


    private ForegroundService(){}


    /**
     * Get instance of the infra Preference Manager
     * Note: this class uses the application context initialized in the {@link Infra} class
     */
    public static ForegroundService getInstance() {
        if (Instance == null) {
            synchronized (ForegroundService.class) {
                if (Instance == null) {
                    Instance = new ForegroundService();
                }
            }
        }
        return Instance;
    }


    /**
     * Register a view for the given brand as foreground
     *
     * @param brandId BrandId
     * @param targetId BrandId
     */
    @Override
    public void register(String brandId, String targetId) {
        LPLog.INSTANCE.i(TAG, "register with brand " + brandId);
        boolean needBroadcast = false;
        if (!TextUtils.isEmpty(targetId) && !mForegroundTargets.contains(targetId)) {
            mForegroundTargets.add(targetId);
            needBroadcast = true;
        }
        if (!TextUtils.isEmpty(brandId) && !mForegroundBrands.contains(brandId)) {
            mForegroundBrands.add(brandId);
            needBroadcast = true;
        }
        if(needBroadcast) {
            mForegroundMillis.put(brandId, System.currentTimeMillis());
            Bundle bundle = new Bundle();
            bundle.putString(SCREEN_FOREGROUND_EXTRA_BRAND_ID, brandId);
            bundle.putString(SCREEN_FOREGROUND_EXTRA_TARGET_ID, targetId);
            bundle.putBoolean(SCREEN_FOREGROUND_EXTRA_IS_FOREGROUND, true);
            LocalBroadcast.sendBroadcast(SCREEN_FOREGROUND_ACTION, bundle);
        }
    }

    /**
     * Unregister a view for the given brand as background
     *
     * @param brandId BrandId
     */
    @Override
    public void unregisterBrand(String brandId) {
        LPLog.INSTANCE.i(TAG, "unregister with brand " + brandId);
        if (!TextUtils.isEmpty(brandId) && mForegroundBrands.contains(brandId)) {
            mForegroundBrands.remove(brandId);

            Bundle bundle = new Bundle();
            bundle.putString(SCREEN_FOREGROUND_EXTRA_BRAND_ID, brandId);
            bundle.putBoolean(SCREEN_FOREGROUND_EXTRA_IS_FOREGROUND, false);
            LocalBroadcast.sendBroadcast(SCREEN_FOREGROUND_ACTION, bundle);
        }
    }

    @Override
    public void unregisterAll() {
        for(String brand : mForegroundBrands){
            unregisterBrand(brand);
        }
        for(String target : mForegroundTargets){
            unregisterBrand(target);
        }
        mForegroundBrands.clear();

    }

    /**
     * Unregister a view for the given brand as background
     *
     * @param targetId BrandId
     */
    @Override
    public void unregisterTarget(String targetId) {
	    LPLog.INSTANCE.i(TAG, "unregister with target " + targetId);
        if (!TextUtils.isEmpty(targetId)) {
            mForegroundTargets.remove(targetId);
        }
    }


    @Override
    public boolean isBrandForeground() {
        return (!mForegroundBrands.isEmpty());
    }

    @Override
    public boolean isBrandForeground(String brandId) {
        return mForegroundBrands.contains(brandId);
    }

    @Override
    public boolean isTargetForeground(String brandId) {
        return mForegroundTargets.contains(brandId);
    }

    @Override
    public void shutDown() {
        unregisterAll();
    }

    /**
     * Method used to determine when messaging app was in
     * foreground last time.
     * @param brandId required brand id
     * @return timestamp of last time when app was foreground or
     * -1 when app was always in background.
     */
    public long getLatestForegroundTime(String brandId) {
        Long lastTime = mForegroundMillis.get(brandId);
        return lastTime != null ? lastTime : -1;
    }
}
