package com.liveperson.infra.otel

import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.otel.exporters.OtlpExporter
import com.liveperson.infra.otel.models.OtlpAttribute
import com.liveperson.infra.otel.models.OtlpSpan
import java.util.Collections
import java.util.concurrent.ConcurrentHashMap

class LPTraceSpanCollector(
    private var exporters: List<OtlpExporter> = emptyList(),
    private val autoExportThreshold: Int = -1
): LPTraceSpanDelegate {

    // completed spans
    private val completedSpans: MutableList<LPTraceSpan> = Collections.synchronizedList(
        mutableListOf()
    )
    // in-progress spans
    private var spansMap: MutableMap<String, LPTraceSpan> = ConcurrentHashMap()

    var activeSpan: LPTraceSpan? = null

    @JvmOverloads
    fun begin(
        dataType: LPTraceType,
        attributes: List<OtlpAttribute> = emptyList()
    ): LPTraceSpan {
        val traceId: String
        var parentSpanId: String? = null
        if (activeSpan != null) {
            traceId = activeSpan!!.traceId
            parentSpanId = activeSpan!!.spanId
        } else {
            traceId = OtelUtils.generateTraceId()
        }

        val spanId = OtelUtils.generateSpanId()

        val span = LPTraceSpan(
            dataType.value,
            OtelUtils.getNanoTime(),
            spanId,
            traceId,
            null,
            3,
            parentSpanId,
            attributes,
            this
        )
        spansMap[span.spanId] = span

        d(TAG, "begin span has id = $spanId")
        return span
    }

    override fun end(id: String) {
        val span = spansMap[id]
        if (span == null || span.endTimeUnixNano != null) {
            return
        }
        span.endTimeUnixNano = OtelUtils.getNanoTime()
        completedSpans.add(span)
        d(TAG, "ended span has id = ${span.spanId}")
        if (autoExportThreshold > 0 && completedSpans.size > autoExportThreshold) {
            export()
        }
        spansMap.remove(id)
    }

    override fun cancel(id: String) {
        spansMap.remove(id)
    }

    /**
     * Get LPTraceSpan by traceType from in-progress spansMap.
     * If there are more than one span of same traceType, get the oldest.
     */
    fun getSpanByTraceType(traceType: LPTraceType): LPTraceSpan? {
        return spansMap.values.asSequence()
            .filter { lpTraceSpan -> lpTraceSpan.name == traceType.value }
            .minByOrNull { it.startTimeUnixNano  }
    }

    fun setExporters(exporters: List<OtlpExporter>) {
        this.exporters = exporters
    }

    fun export() {
        if (exporters.isEmpty() || completedSpans.isEmpty()) {
            return
        }

        val spans = completedSpans.toTypedArray().map {
            toOtlpSpan(it)
        }
        completedSpans.clear()
        exporters.forEach {
            it.export(spans)
        }
    }

    fun flush() {
        spansMap.clear()
        completedSpans.clear()
    }

    fun endActiveSpan() {
        activeSpan?.end()
        activeSpan = null
    }

    private fun toOtlpSpan(lpSpan: LPTraceSpan): OtlpSpan {
        return OtlpSpan(
            lpSpan.name,
            lpSpan.startTimeUnixNano,
            lpSpan.kind,
            lpSpan.spanId,
            lpSpan.traceId,
            lpSpan.endTimeUnixNano,
            lpSpan.parentSpanId,
            lpSpan.attributes
        )
    }

    companion object {
        private const val TAG = "LPTraceSpanCollector"
    }
}
