package com.liveperson.infra.network.http.request;


import android.util.Pair;

import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.http.HttpUtilsKt;
import com.liveperson.infra.network.http.body.HttpRequestBody;
import com.liveperson.infra.otel.LPTraceType;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import okhttp3.Response;


/**
 * Created by ofira on 03/11/2015.
 */
public abstract class HttpRequest {

    private static final String TAG = "HttpRequest";
    private static final int DEFAULT_TIMEOUT = 30000;
    private int mTimeout = DEFAULT_TIMEOUT;
    private String mUrl;
    private HttpMethod mMethod;
    private List<Pair<String, String>> mHeaders;
    protected HttpRequestBody mRequestBody;
    protected ICallback mCallback;
    private final List<String> mCertificatePinningKeys;
    private static final String USER_AGENT = "User-Agent";
    private static final String ANDROID = "ANDROID";

    protected LPTraceType mTraceType;


    public enum HttpMethod {GET, HEAD, POST, PUT, DELETE}

    public HttpRequest(String url, HttpMethod method, LPTraceType traceType) {
        mUrl = url;
        mMethod = method;
        mHeaders = new ArrayList<>();
        // adding a default User-Agent header with ANDROID
        mHeaders.add(Pair.create(USER_AGENT, ANDROID));
        mTraceType = traceType;
        mCertificatePinningKeys = new ArrayList<>();
    }

    public void processResponse(Response response) {
        try {
            if (isResponseSuccessful(response)) {
                LPLog.INSTANCE.i(TAG, "onSuccess! " + response.code());
                if (mCallback != null) {
                    reportSuccess(response);
                }
            } else {
                LPLog.INSTANCE.i(TAG, "onResponseFailure " + response);
                if (mCallback != null) {
                    Exception exception = HttpUtilsKt.asException(response);
                    processErrorResponse(exception);
                }
            }

        } catch (IOException e) {
            processErrorResponse(e);
        }
    }

    private boolean isResponseSuccessful(Response response) {
        return response != null && response.isSuccessful();
    }

    public void processErrorResponse(Exception e) {
        if (e != null) {
            LPLog.INSTANCE.w(TAG, " Error while sending http request : Error msg: " + e.getMessage(), e);
        } else {
            LPLog.INSTANCE.w(TAG, " Error while sending http request; no exception given");
        }

        if (mCallback != null) {
            mCallback.onError(e);
        }
    }

    protected void reportSuccess(Response response) throws IOException {
        mCallback.onSuccess(response == null ? "" : response.body().string());
    }

    public abstract HttpRequestBody getRequestBody();

    public abstract void setBody(HttpRequestBody body);

    // check if we are adding User-Agent header if so we want to set the already added default header to our new one. otherwise just add the header.
    public void addHeader(String name, String value) {
        if(name.equals(USER_AGENT)){
            mHeaders.set(0, Pair.create(USER_AGENT, value));
        } else {
            mHeaders.add(Pair.create(name, value));
        }
    }

    public List<Pair<String, String>> getHeaders() {
        return mHeaders;
    }

    public String getUrl() {
        return mUrl;
    }

    public void setUrl(String mUrl) {
        this.mUrl = mUrl;
    }

    public HttpMethod getMethod() {
        return mMethod;
    }

    public List<String> getCertificatePinningKeys() {
        return mCertificatePinningKeys;
    }

    public int getTimeout() {
        return mTimeout;
    }

    public void setTimeout(int timeout) {
        mTimeout = timeout;
    }

    public ICallback getCallback() {
        return mCallback;
    }

    public void setCertificatePinningKeys(List<String> certificatePinningKeys) {
        mCertificatePinningKeys.clear();
        if (certificatePinningKeys != null && !certificatePinningKeys.isEmpty()) {
            mCertificatePinningKeys.addAll(certificatePinningKeys);
        }
    }

    public void setCallback(ICallback callback) {
        this.mCallback = callback;
    }

    public LPTraceType getTraceType() {
        return mTraceType;
    }

    public void setTraceType(LPTraceType traceType) {
        this.mTraceType = traceType;
    }
}
