package com.liveperson.infra.otel

import android.os.Build
import com.liveperson.infra.BuildConfig
import com.liveperson.infra.CampaignInfo
import com.liveperson.infra.otel.LPTelemetryConstants.INSTRUMENTATION_SCOPE_NAME
import com.liveperson.infra.otel.LPTelemetryConstants.SDK_LANGUAGE
import com.liveperson.infra.otel.LPTelemetryConstants.SERVICE_NAME
import com.liveperson.infra.otel.exporters.OtlpExporter
import com.liveperson.infra.otel.exporters.OtlpHttpExporter
import com.liveperson.infra.otel.models.OtlpAttribute
import com.liveperson.infra.otel.models.OtlpResource
import com.liveperson.infra.otel.models.OtlpValueData
import kotlin.random.Random

object LPTelemetryManager {
    private const val TAG = "LPTelemetryManager"
    private const val OTLP_COLLECTOR_ENDPOINT_FORMAT = "https://%s/v1/traces"
    private const val AUTO_EXPORT_THRESHOLD = 5
    /**
     if the domain is "empty.liveperson.net" then disable exporting data
     **/
    private const val DEFAULT_OTEL_DOMAIN = "empty.liveperson.net"

    private val RANDOM_SAMPLING_RATE = Random.nextInt(1,100)
    private var defaultCollector: LPTraceSpanCollector? = null
    private var parentCollector: LPTraceSpanCollector? = null
    private var activeCollector: LPTraceSpanCollector? = parentCollector ?: defaultCollector
    private var exporters: List<OtlpExporter> = emptyList()
    private lateinit var mOtlpResource: OtlpResource
    private var isEnable: Boolean = true

    @JvmStatic
    fun init(accountId: String) {
        mOtlpResource = initResource(accountId)
        defaultCollector = LPTraceSpanCollector(autoExportThreshold = AUTO_EXPORT_THRESHOLD)
    }

    private fun initResource(accountId: String): OtlpResource {
        return OtlpResource(
            listOf(
                OtlpAttribute(LPTelemetryAttributeKey.SERVICE_NAME.value, OtlpValueData.StringValue(SERVICE_NAME)),
                OtlpAttribute(LPTelemetryAttributeKey.TELEMETRY_SDK_LANGUAGE.value, OtlpValueData.StringValue(SDK_LANGUAGE)),
                OtlpAttribute(LPTelemetryAttributeKey.TELEMETRY_SDK_NAME.value, OtlpValueData.StringValue(INSTRUMENTATION_SCOPE_NAME)),
                OtlpAttribute(LPTelemetryAttributeKey.OS_TYPE.value, OtlpValueData.StringValue("Android")),
                OtlpAttribute(LPTelemetryAttributeKey.OS_VERSION.value, OtlpValueData.StringValue(Build.VERSION.RELEASE)),
                OtlpAttribute(LPTelemetryAttributeKey.TELEMETRY_SDK_VERSION.value, OtlpValueData.StringValue(BuildConfig.VERSION_NAME)),
                OtlpAttribute(LPTelemetryAttributeKey.DEVICE_MODEL_IDENTIFIER.value, OtlpValueData.StringValue("${Build.MANUFACTURER},${Build.MODEL}")),
                OtlpAttribute(LPTelemetryAttributeKey.LP_ACCOUNT_ID.value, OtlpValueData.StringValue(accountId)),
            )
        )
    }

    fun prepareParentCollector(dataType: LPTraceType = LPTraceType.SHOW_CONV): LPTraceSpan {
        val collector = LPTraceSpanCollector(exporters)
        val span = collector.begin(dataType)
        collector.activeSpan = span
        parentCollector = collector
        return span
    }

    fun endParentCollector(otelFullUrl: String) {
        setExporter(otelFullUrl)
        parentCollector?.endActiveSpan()
        parentCollector?.export()
        parentCollector = null
    }

    /**
     * init showConversation - a parent span
     */
    fun initShowConversationSpan(brandId: String, campaignInfo: CampaignInfo?) {
        val showConversationSpan = prepareParentCollector(LPTraceType.SHOW_CONV)
        showConversationSpan.addAttribute(LPTelemetryAttributeKey.LP_ACCOUNT_ID, brandId)
        if (campaignInfo != null) {
            showConversationSpan.addAttribute(
                LPTelemetryAttributeKey.LP_CAMPAIGN_ID,
                campaignInfo.campaignId?.toString()
            )
            showConversationSpan.addAttribute(
                LPTelemetryAttributeKey.LP_ENGAGEMENT_ID,
                campaignInfo.engagementId?.toString()
            )
            showConversationSpan.addAttribute(
                LPTelemetryAttributeKey.LP_SESSION_ID,
                campaignInfo.sessionId
            )
            showConversationSpan.addAttribute(
                LPTelemetryAttributeKey.LP_VISITOR_ID,
                campaignInfo.visitorId
            )
            showConversationSpan.addAttribute(
                LPTelemetryAttributeKey.LP_CONTEXT_ID,
                campaignInfo.contextId
            )
        }
    }

    @JvmOverloads
    fun begin(dataType: LPTraceType, attributes: List<OtlpAttribute> = emptyList()): LPTraceSpan? {
        if (LPTraceType.NOT_TRACKING == dataType) {
            return null
        }
        activeCollector = parentCollector ?: defaultCollector
        return activeCollector?.begin(dataType, attributes)
    }

    /**
     * Get LPTraceSpan by traceType in active collector
     */
    fun getSpanByTraceType(traceType: LPTraceType): LPTraceSpan? {
        return activeCollector?.getSpanByTraceType(traceType)
    }

    private fun enable() {
        // enable
        isEnable = true
    }

    private fun disable() {
        // disable
        isEnable = false
        defaultCollector = null
        parentCollector = null
    }

    private fun setExporter(otelFullUrl: String) {
        val domain = OtelUtils.extractDomain(otelFullUrl)
        val samplingRate = OtelUtils.extractSamplingFromDomain(otelFullUrl)
        if (RANDOM_SAMPLING_RATE > samplingRate) {
            disable()
        } else {
            enable()
        }

        if (!isEnable || domain.isEmpty() || domain.equals(DEFAULT_OTEL_DOMAIN, true)) {
            return
        }

        // prevent leaking init
        if (!::mOtlpResource.isInitialized) {
            return
        }

        val exporter = OtlpHttpExporter(String.format(OTLP_COLLECTOR_ENDPOINT_FORMAT, domain), mOtlpResource)
        defaultCollector?.setExporters(listOf(exporter))
        parentCollector?.setExporters(listOf(exporter))
    }

    fun export(otelFullUrl: String) {
        setExporter(otelFullUrl)
        if (!isEnable) {
            return
        }
        defaultCollector?.export()
        parentCollector?.endActiveSpan()
        parentCollector?.export()
        defaultCollector?.flush()
        parentCollector?.flush()
    }
}
