package com.liveperson.infra.otel

import com.liveperson.infra.otel.models.OtlpResource
import com.liveperson.infra.otel.models.OtlpResourceSpan
import com.liveperson.infra.otel.models.OtlpScope
import com.liveperson.infra.otel.models.OtlpScopeSpan
import com.liveperson.infra.otel.models.OtlpSpan
import com.liveperson.infra.otel.models.OtlpTraceSpan
import java.util.Random
import java.util.concurrent.TimeUnit

object OtelUtils {

    /**
     * Default kind of span sent from mobile sdk(ios and android).
     */
    const val SPAN_KIND_CLIENT = 3

    fun generateTraceId(): String {
        return generateRandomHexString(32)
    }

    fun generateSpanId(): String {
        return generateRandomHexString(16)
    }

    private fun generateRandomHexString(length: Int): String {
        val randomBytes = ByteArray(length / 2)
        val random = Random()
        random.nextBytes(randomBytes)
        return randomBytes.joinToString("") { "%02x".format(it) }
    }

    fun getNanoTime(): Long {
        return TimeUnit.MILLISECONDS.toNanos(System.currentTimeMillis())
    }

    /**
     * Extract sampling from domain if it's there.
     * Return 0 if it's not.
     * Example:
     *  - site.com/10 -> 10
     *  - site.com/text -> 0
     *  - site.net/101 -> 0
     */
    fun extractSamplingFromDomain(domain: String): Int {
        val regex = Regex("/(\\d{1,3})")
        val matchResult = regex.find(domain)
        return matchResult?.groupValues?.getOrNull(1)?.toIntOrNull()?.takeIf { it in 1..100 } ?: 0
    }

    /**
     * Extract domain from url where it may contain http(s):// or not
     * Example:
     *  - sub.liveperson.net/90
     *  - https://sub.liveperson.net/10
     */
    fun extractDomain(url: String): String {
        // where (https?://)? makes the "http://" or "https://" part optional, and ([^/]+) captures the domain
        val regex = Regex("(https?://)?([^/]+)")
        val matchResult = regex.find(url)
        System.currentTimeMillis()
        return matchResult?.groupValues?.getOrNull(2) ?: ""
    }

    fun createTrace(resource: OtlpResource, sortedSpans: List<OtlpSpan>): OtlpTraceSpan {
        val scope = OtlpScope(
            LPTelemetryConstants.INSTRUMENTATION_SCOPE_NAME,
            LPTelemetryConstants.INSTRUMENTATION_SCOPE_VERSION
        )
        val scopeSpans = listOf(OtlpScopeSpan(scope, sortedSpans))
        val resourceSpan = OtlpResourceSpan(scopeSpans, resource)

        return OtlpTraceSpan(listOf(resourceSpan))
    }
}
