package com.liveperson.infra.utils

import androidx.annotation.VisibleForTesting

private const val BEGINNING = 0

private val PHONE_INCORRECT_CHARACTERS = setOf('+', '*', '#', '/')
private val URL_INCORRECT_CHARACTERS = setOf('@')


/**
 * Method used to validate possible link whether it contains
 * only appropriate characters.
 *
 * @param index - index of character required to be validated
 * @param set - set of invald characters
 *
 * @return true if index is out of char sequence length (negative or bigger than length)
 * or if requested by index character isn't mentioned in [set].
 */
@VisibleForTesting
private fun CharSequence.hasCorrectCharAtIndex(index: Int, set: Set<Char>): Boolean {
    return if (index < BEGINNING || index > lastIndex) {
        true
    } else {
        !set.contains(this[index])
    }
}

/**
 * Method used to validate possible phone link whether it contains
 * only appropriate characters.
 *
 * @param index - index of character required to be validated
 *
 * @return true if index is out of char sequence length (negative or bigger than length)
 * or if requested by index character isn't mentioned in [PHONE_INCORRECT_CHARACTERS]
 */
fun CharSequence.phoneHasCorrectCharAtIndex(index: Int): Boolean {
    return hasCorrectCharAtIndex(index, PHONE_INCORRECT_CHARACTERS)
}

/**
 * Method used to validate possible url link whether it contains
 * only appropriate characters.
 *
 * @param index - index of character required to be validated
 *
 * @return true if index is out of char sequence length (negative or bigger than length)
 * or if requested by index character isn't mentioned in [URL_INCORRECT_CHARACTERS]
 */
fun CharSequence.urlHasCorrectCharAtIndex(index: Int): Boolean {
    return hasCorrectCharAtIndex(index, URL_INCORRECT_CHARACTERS)
}