package com.liveperson.infra.analytics.events
import android.content.Context
import com.liveperson.infra.analytics.*
import com.liveperson.infra.auth.LPAuthenticationType

/**
 * Class to gather LivePerson facade's analytics data.
 */
open class LivePersonFacade(private val analyticsService: AnalyticsService) {

    companion object {
        private const val API_VERSION = "api_version"
        private const val STARTUP_TIME = "startup_time"
        private const val EXECUTION_TIME = "execution_time"
        private const val AUTH_TYPE = "auth_type"
        private const val HAS_CAMPAIGN_INFO = "has_campaign_info"
        private const val HAS_NOTIFICATION_TOKEN = "has_notification_token"
        private const val HAS_APP_ID = "has_app_id"
        private const val SHOW_NOTIFICATION = "show_notification"
        private const val Error = "error"
    }

    /**
     * Gather analytics data for LivePerson facade's initialize method.
     * @param version method version to track If deprecated one is still in use
     * @param startUpTime Time it took to complete initialization process
     * @param error Error If failed to initialize. Null otherwise
     */
    fun trackInit(version: LPAPIVersion, startUpTime: Long, error: String?) {
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.INITIALIZE,
                EventProperty(API_VERSION, version.value),
                EventProperty(STARTUP_TIME, startUpTime),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's showConversation method.
     * @param version method version to track If deprecated one is still in use
     * @param authType Type of authentication being used by customers
     * @param hasCampaignInfo Is customer passing in campaign info
     * @param error Error If failed to show. Null otherwise
     */
    fun trackDisplayConversation(version: LPAPIVersion, authType: LPAuthenticationType?, hasCampaignInfo: Boolean, isUsingFragment: Boolean, error: String?) {
        val eventName = if (isUsingFragment) EventName.LivePerson.SHOW_CONVERSATION_HOST_CONTROL else EventName.LivePerson.SHOW_CONVERSATION_LP_CONTROL
        analyticsService.logUserEvent(AnalyticsEvent(eventName,
                EventProperty(API_VERSION, version.value),
                EventProperty(AUTH_TYPE, authType),
                EventProperty(HAS_CAMPAIGN_INFO, hasCampaignInfo),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's registerLpPusher method.
     * @param version method version to track If deprecated one is still in use
     * @param authType Type of authentication being used by customers
     * @param isPushTokenProvided Is push notification/gcm token provided by host app
     * @param error Error If failed to register. Null otherwise
     */
    fun trackRegisterLPPusher(version: LPAPIVersion, brandId: String?, authType: LPAuthenticationType?, isPushTokenProvided: Boolean, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.REGISTER_PUSHER,
                EventProperty(API_VERSION, version.value),
                EventProperty(AUTH_TYPE, authType),
                EventProperty(HAS_NOTIFICATION_TOKEN, isPushTokenProvided),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's getUnreadMessageCount method.
     * @param version method version to track If deprecated one is still in use
     * @param authType Type of authentication being used by customers
     * @param error Error If failed to get count. Null otherwise
     */
    fun trackGetUnreadMessageCount(version: LPAPIVersion, authType: LPAuthenticationType?, error: String?) {
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.GET_UNREAD_MESSAGES_COUNT,
                EventProperty(API_VERSION, version.value),
                EventProperty(AUTH_TYPE, authType),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's reconnect method.
     * @param version method version to track If deprecated one is still in use
     * @param authType Type of authentication being used by customers
     * @param error Error If failed to reconnect. Null otherwise
     */
    fun trackReconnect(version: LPAPIVersion, authType: LPAuthenticationType?, error: String?) {
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.RECONNECT,
                EventProperty(API_VERSION, version.value),
                EventProperty(AUTH_TYPE, authType),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's logout method.
     * @param version method version to track If deprecated one is still in use
     * @param hasAppId If host app has provided appId during logout
     * @param error Error If failed to logout. Null otherwise
     */
    fun trackLogout(version: LPAPIVersion, brandId: String?, executionTime: Long, hasAppId: Boolean, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.LOGOUT,
                EventProperty(API_VERSION, version.value),
                EventProperty(EXECUTION_TIME, executionTime),
                EventProperty(HAS_APP_ID, hasAppId),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's handlePush method.
     * @param version method version to track If deprecated one is still in use
     * @param brandId Brand's account id
     * @param showNotification If host app has given permission to SDK to display notification
     * @param error Error If failed handle. Null otherwise
     */
    fun trackHandlePushMessage(version: LPAPIVersion, appContext: Context?, brandId: String?, showNotification: Boolean, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.setAppContext(appContext)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.HANDLE_PUSH,
                EventProperty(API_VERSION, version.value),
                EventProperty(SHOW_NOTIFICATION, showNotification),
                EventProperty(Error, error)))
    }
}