package com.liveperson.infra.model;

import android.os.Parcel;
import android.os.Parcelable;
import androidx.annotation.NonNull;
import android.text.TextUtils;

import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.utils.EncryptionVersion;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

import static com.liveperson.infra.errors.ErrorCode.ERR_0000003F;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000040;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000041;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000042;

/**
 * Created by Yanda on 04/23/2019.
 * <p>
 * Welcome message class with quick reply support.
 */
public final class LPWelcomeMessage implements Parcelable {

    private static final String TAG = "LPWelcomeMessage";
    private static final int MAX_ITEMS = 24;
    private static final int MAX_MESSAGE_OPTION_LENGTH = 25;
    private static final int MIN_MESSAGE_OPTION_LENGTH = 1;
    private static final int DEFAULT_NUMBER_OF_ITEMS_PER_ROW = 8;
    private static final int WELCOME_MSG_SEQUENCE_NUMBER = -4;
    private static final String KEY_TYPE = "type";
    private static final String KEY_TOOLTIP = "tooltip";
    private static final String KEY_TITLE = "title";
    private static final String KEY_TEXT = "text";
    private static final String KEY_ACTIONS = "actions";
    private static final String KEY_METADATA = "metadata";
    private static final String KEY_CLICK = "click";
    private static final String KEY_ITEMS_PER_ROW = "itemsPerRow";
    private static final String KEY_REPLIES = "replies";
    private static final String KEY_QUICK_REPLIES = "quick_replies_key";
    private static final String KEY_TIME_STAMP = "timestamp_key";
    private static final String KEY_ORIGINATOR_ID = "originator_id_key";
    private static final String KEY_SEQUENCE = "sequence_key";
    private static final String KEY_SHOW = "show_key";
    private static final String VALUE_BUTTON = "button";
    private static final String VALUE_PUBLISH_TEXT = "publishText";
    private static final String VALUE_QUICK_REPLIES = "quickReplies";
    private static final String VALUE_ORIGINATOR_ID = "originatorId";

    private String welcomeMessage;
    private int numberOfItemsPerRow;
    private List<MessageOption> messageOptions;
    private MessageFrequency messageFrequency;

    /**
     * @param welcomeMessage Welcome message text.
     */
    public LPWelcomeMessage(String welcomeMessage) {
        this.welcomeMessage = welcomeMessage;
        numberOfItemsPerRow = DEFAULT_NUMBER_OF_ITEMS_PER_ROW;
        messageOptions = new ArrayList<>();
        messageFrequency = MessageFrequency.FIRST_TIME_CONVERSATION;
    }

    public String getWelcomeMessage() {
        return welcomeMessage;
    }

    /**
     * If welcome message is null, use the welcome message in string resources.
     *
     * @return If default welcome message should be used.
     */
    public boolean shouldUseDefaultMessage() {
        return TextUtils.isEmpty(welcomeMessage);
    }

    /**
     * Set how many quick reply items should be displayed per show.
     *
     * @param numberOfItemsPerRow Number of items per row.
     */
    public void setNumberOfItemsPerRow(int numberOfItemsPerRow) {
        this.numberOfItemsPerRow = numberOfItemsPerRow;
    }

    public void setMessageOptions(@NonNull List<MessageOption> messageOptions) throws Exception {
        if (messageOptions.size() > MAX_ITEMS) {
            String message = "Maximum number of quick replies is " + MAX_ITEMS;
            LPLog.INSTANCE.e(TAG, ERR_0000003F, message);
            throw new Exception(message);
        }
        for (MessageOption item : messageOptions) {
            if (item.getDisplayText().length() > MAX_MESSAGE_OPTION_LENGTH) {
                String message = "Maximum number of characters is " + MAX_MESSAGE_OPTION_LENGTH;
                LPLog.INSTANCE.e(TAG, ERR_00000040, message);
                throw new Exception(message);
            } else if (item.getDisplayText().trim().length() < MIN_MESSAGE_OPTION_LENGTH) {
                String message = "Minimum number of characters is " + MIN_MESSAGE_OPTION_LENGTH;
                LPLog.INSTANCE.e(TAG, ERR_00000041, message);
                throw new Exception(message);
            } else {
                this.messageOptions.add(item);
            }
            if (this.messageOptions.size() == MAX_ITEMS) {
                return;
            }
        }
    }

    public void setMessageFrequency(MessageFrequency messageFrequency) {
        this.messageFrequency = messageFrequency;
    }

    /**
     * @return Show welcome message only for first conversation or for each conversation. Default is first conversation only.
     */
    public MessageFrequency getMessageFrequency() {
        return messageFrequency;
    }

    /**
     * Build json string, which is used for quick reply buttons.
     *
     * @return Json string.
     */
    public String getQuickReplies(boolean buttonClickable) {
        try {
            JSONArray repliesJsonArray = new JSONArray();
            for (MessageOption item : messageOptions) {
                JSONObject itemJsonObject = new JSONObject();
                itemJsonObject.put(KEY_TYPE, VALUE_BUTTON);
                itemJsonObject.put(KEY_TOOLTIP, item.getDisplayText());
                itemJsonObject.put(KEY_TITLE, item.getDisplayText());
                JSONObject clickJsonObject = new JSONObject();
                if (buttonClickable) {
                    JSONArray actionJsonArray = new JSONArray();
                    JSONObject actionJsonObject = new JSONObject();
                    actionJsonObject.put(KEY_TYPE, VALUE_PUBLISH_TEXT);
                    actionJsonObject.put(KEY_TEXT, item.getValue());
                    actionJsonArray.put(actionJsonObject);
                    clickJsonObject.put(KEY_ACTIONS, actionJsonArray);
                    if (item.getMetadata() != null) {
                        clickJsonObject.put(KEY_METADATA, item.getMetadata());
                    }
                }
                itemJsonObject.put(KEY_CLICK, clickJsonObject);
                repliesJsonArray.put(itemJsonObject);
            }

            JSONObject quickRepliesJsonObject = new JSONObject();
            quickRepliesJsonObject.put(KEY_TYPE, VALUE_QUICK_REPLIES);
            quickRepliesJsonObject.put(KEY_ITEMS_PER_ROW, numberOfItemsPerRow);
            quickRepliesJsonObject.put(KEY_REPLIES, repliesJsonArray);

            JSONObject jsonObject = new JSONObject();
            jsonObject.put(KEY_QUICK_REPLIES, quickRepliesJsonObject);
            jsonObject.put(KEY_TIME_STAMP, System.currentTimeMillis());
            jsonObject.put(KEY_ORIGINATOR_ID, VALUE_ORIGINATOR_ID);
            jsonObject.put(KEY_SEQUENCE, WELCOME_MSG_SEQUENCE_NUMBER);
            jsonObject.put(KEY_SHOW, true);

            return DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, jsonObject.toString());

        } catch (JSONException e) {
            LPLog.INSTANCE.e(TAG, ERR_00000042, "Error when parsing json for welcome message quick replies", e);
            return "";
        }
    }

    private LPWelcomeMessage(Parcel in) {
        welcomeMessage = in.readString();
        numberOfItemsPerRow = in.readInt();
        if(messageOptions == null) {
            messageOptions = new ArrayList<>();
        } else {
            messageOptions.clear();
        }
        in.readList(messageOptions, LPWelcomeMessage.class.getClassLoader());
        messageFrequency = MessageFrequency.values()[in.readInt()];
    }

    public static final Creator CREATOR = new Creator() {
        @Override
        public LPWelcomeMessage createFromParcel(Parcel in) {
            return new LPWelcomeMessage(in);
        }

        @Override
        public LPWelcomeMessage[] newArray(int size) {
            return new LPWelcomeMessage[size];
        }
    };

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(welcomeMessage);
        dest.writeInt(numberOfItemsPerRow);
        dest.writeList(messageOptions);
        dest.writeInt(messageFrequency.ordinal());
    }

    /**
     * The frequency to display welcome message.
     */
    public enum MessageFrequency {
        FIRST_TIME_CONVERSATION, EVERY_CONVERSATION
    }
}
