package com.liveperson.infra.utils;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.liveperson.infra.Infra;
import com.liveperson.infra.log.LPLog;

/**
 * Created by nirni on 12/6/16.
 */

public class VersionUtils {

    private static final String TAG = "VersionUtils";

    /**
     * Check if the host version (current SDK version) is valid against the given minSdkVersion.
     * minSdkVersion should be received from one of LP server as the minimum version.
     *
     * @param minSdkVersion
     * @return true - if the current SDK version is smaller than the given minSdkVersion, false - otherwise
     */
    public static boolean isValidSdkVersion(String minSdkVersion) {

        String hostVersion = Infra.instance.getHostVersion();

        if (versionCompare(hostVersion, minSdkVersion) < 0) {

            LPLog.INSTANCE.w(TAG, "isValidSdkVersion: Current SDK version (" + hostVersion + ") is smaller than the one from the server (" + minSdkVersion + "). ");
            return false;
        }

        LPLog.INSTANCE.d(TAG, "isValidSdkVersion: Version check is OK");
        return true;
    }

    /**
     * Compares two version strings.
     * <p/>
     * Use this instead of String.compareTo() for a non-lexicographical
     * comparison that works for version strings. e.g. "1.10".compareTo("1.6").
     *
     * @param str1 a string of ordinal numbers separated by decimal points.
     * @param str2 a string of ordinal numbers separated by decimal points.
     * @return The result is a negative integer if str1 is _numerically_ less than str2.
     * The result is a positive integer if str1 is _numerically_ greater than str2.
     * The result is zero if the strings are _numerically_ equal.
     * @note It does not work if "1.10" is supposed to be equal to "1.10.0".
     */
    private static Integer versionCompare(String str1, String str2) throws NullPointerException {

        LPLog.INSTANCE.d(TAG, "versionCompare str1 " + str1);
        LPLog.INSTANCE.d(TAG, "versionCompare str2 " + str2);

        // If one of the version strings is null return 0 (equal)
        if (TextUtils.isEmpty(str1) || TextUtils.isEmpty(str2)) {
            LPLog.INSTANCE.w(TAG, "versionCompare: one of the compared version is null or empty");
            return 0;
        }

        String[] values1 = str1.split("\\.");
        String[] values2 = str2.split("\\.");

        int i = 0;
        // set index to first non-equal ordinal or length of shortest version string
        while (i < values1.length && i < values2.length && values1[i].equals(values2[i])) {
            i++;
        }

        // compare first non-equal ordinal number
        if (i < values1.length && i < values2.length) {
            int diff = Integer.valueOf(values1[i]).compareTo(Integer.valueOf(values2[i]));
            return Integer.signum(diff);
        }
        // the strings are equal or one string is a substring of the other
        // e.g. "1.2.3" = "1.2.3" or "1.2.3" < "1.2.3.4"
        else {
            return Integer.signum(values1.length - values2.length);
        }
    }

    /**
     * Return app version from provided context
     * @param appContext application context
     * @return
     */
    @NonNull
    public static String getAppVersion(@NonNull Context appContext) {
        try {
            PackageInfo packageInfo = appContext.getPackageManager().getPackageInfo(appContext.getPackageName(), 0);
            return packageInfo.versionName;
        } catch (PackageManager.NameNotFoundException nnfe) {
            LPLog.INSTANCE.w(TAG, "Could not find package for our own package name!", nnfe);
        } catch (NullPointerException npe) {
            LPLog.INSTANCE.w(TAG, "NPE while trying to fetch our app's version number!", npe);
        }
        return "";
    }
}
