package com.liveperson.infra.analytics.support

import android.app.Activity
import android.content.Context
import com.liveperson.infra.Infra
import com.liveperson.infra.analytics.*
import com.liveperson.infra.auth.LPAuthenticationParams
import com.liveperson.infra.auth.LPAuthenticationType
import com.liveperson.infra.log.LogLevel
import com.liveperson.infra.utils.AnalyticsUtils


open class LPAnalyticsFacadeWrapper private constructor() {

    private object HOLDER {
        val INSTANCE = LPAnalyticsFacadeWrapper()

    }

    companion object {
        val instance: LPAnalyticsFacadeWrapper by lazy { HOLDER.INSTANCE }
    }

    /**
     * Send analytics data of SDK initialization state. If initialized successfully, error will be empty
     * @param version method version
     * @param startTime Time when initialization started
     * @param error Error If initialization failed, null otherwise
     */
    open fun trackInit(version: LPAPIVersion, startTime: Long, error: String?) {
        LPAnalytics.LivePerson.trackInit(version, AnalyticsUtils.calculateExecutionTime(startTime), error)
    }


    /**
     * Send analytics data of show conversation api usage. If SDK initialized successfully, error will be empty
     * @param version method version
     * @param lpAuthenticationParams Authentication parameters provided
     * @param hasCampaignInfo If host app has provided campaign info
     * @param isUsingFragment If host app is using activity or fragment
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackDisplayConversation(version: LPAPIVersion, lpAuthenticationParams: LPAuthenticationParams?, hasCampaignInfo: Boolean, isUsingFragment: Boolean, error: String?) {
        val authType = lpAuthenticationParams?.authType
        LPAnalytics.LivePerson.trackDisplayConversation(version, authType, hasCampaignInfo, isUsingFragment, error)
    }

    /**
     * Send analytics data of register pusher api usage. If SDK initialized successfully, error will be empty
     * @param version method version
     * @param brandId brand's account id
     * @param lpAuthenticationParams Authentication parameters provided
     * @param isPushTokenProvided If host app has provided device token
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackRegisterLPPusher(version: LPAPIVersion, brandId: String, lpAuthenticationParams: LPAuthenticationParams?, isPushTokenProvided: Boolean, error: String?) {
        val authType = lpAuthenticationParams?.authType
        LPAnalytics.LivePerson.trackRegisterLPPusher(version, brandId, authType, isPushTokenProvided, error)
    }

    /**
     * Send analytics data of get unread messages count api usage. If SDK initialized successfully, error will be empty
     * @param version method version
     * @param lpAuthenticationParams Authentication parameters provided
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackGetUnreadMessageCount(version: LPAPIVersion,brandId: String?, hasAppId: Boolean,lpAuthenticationParams: LPAuthenticationParams?, callBackAvailable: Boolean ,error: String?) {
        val authType = lpAuthenticationParams?.authType
        LPAnalytics.LivePerson.trackGetUnreadMessageCount(version,brandId, hasAppId,authType, callBackAvailable,error)
    }

    /**
     * Send analytics data of reconnect api usage. If SDK initialized successfully, error will be empty
     * @param version method version
     * @param brandId brand's account id
     * @param lpAuthenticationParams Authentication parameters provided
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackReconnect(version: LPAPIVersion,brandId: String? ,lpAuthenticationParams: LPAuthenticationParams?, error: String?) {
        val authType = lpAuthenticationParams?.authType
        LPAnalytics.LivePerson.trackReconnect(version,brandId, authType, error)
    }

    /**
     * Send analytics data of logout api usage. If SDK initialized successfully, error will be empty
     * @param version method version
     * @param brandId brand's account id
     * @param startTime time when logout was called.
     * @param hasAppId Is appId provided
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackLogout(version: LPAPIVersion, brandId: String?, startTime: Long, hasAppId: Boolean, error: String?) {
        LPAnalytics.LivePerson.trackLogout(version, brandId, AnalyticsUtils.calculateExecutionTime(startTime), hasAppId, error)
    }

    /**
     * Send analytics data of handlePushMessage api usage. If brandId found, error will be empty
     * @param version method version
     * @param ctx application context coming from brand app
     * @param brandId brand's account id
     * @param showNotification If SDK should display notification
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackHandlePushMessage(version: LPAPIVersion, ctx: Context?, brandId: String?, showNotification: Boolean, error: String?) {

        var context = ctx
        if (context != null) context = context.applicationContext
        LPAnalytics.LivePerson.trackHandlePushMessage(version, context, brandId, showNotification, error)
    }


    /**
     * Send analytics data of show hideConversation api usage. If SDK initialized successfully, error will be empty
     * @param version method version
     * @param activity Activity parameters : Used to retrieve brandId & LPAuthenticationParams
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackHideConversation(version: LPAPIVersion, activity: Activity?, error: String?) {
        val lpAuthenticationParams: LPAuthenticationParams? = activity?.intent?.getParcelableExtra(Infra.KEY_AUTH_KEY)
        val brandId = activity?.intent?.getStringExtra(Infra.KEY_BRAND_ID)
        LPAnalytics.LivePerson.trackHideConversation(version, lpAuthenticationParams?.authType, brandId, error)
    }

    /**
     * Send analytics data of show conversation api usage. If SDK initialized successfully, error will be empty
     * @param lpAuthenticationParams Authentication parameters provided
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackUpdateTokenInBackground(lpAuthenticationParams: LPAuthenticationParams?, brandId: String?, error: String?) {
        LPAnalytics.LivePerson.trackUpdateTokenInBackground(lpAuthenticationParams?.authType, brandId, error)
    }

    /**
     * Send analytics data for tracking setCallBack & removeCallback api usage. If SDK initialized successfully, error will be empty
     * @param brandId brand's account id
     * @param isListenerAvailable boolean value to check if listener is available for setCallBack method of api
     * @param isSetCallback to check if the track request if for setCallback or removeCallback
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackCallback(brandId: String?, isListenerAvailable:Boolean, error: String?, isSetCallback: Boolean) {

        LPAnalytics.LivePerson.trackSetCallback(brandId, isListenerAvailable,error, isSetCallback)
    }

    /**
     * Send analytics data for tracking setUserProfile api usage. If SDK initialized successfully, error will be empty
     * @param version method version
     * @param brandId brand's account id
     * @param hasAppId Is appId provided
     * @param hasFirstName if user has provided firstName set to true
     * @param hasLastName if user has provided lastName set to true
     * @param hasPhoneNumber if user has provided phone set to true
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackSetUserProfile(version: LPAPIVersion, brandId: String?, hasAppId: Boolean, hasFirstName : Boolean,hasLastName : Boolean,hasPhoneNumber:Boolean,error: String?) {
        LPAnalytics.LivePerson.trackSetUserProfile(version, brandId, hasAppId,hasFirstName,hasLastName,hasPhoneNumber, error)
    }

    /**
     * Send analytics data for tracking checkActiveConversation api usage. If SDK initialized successfully, error will be empty
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackCheckActiveConversation(brandId: String?, error: String?) {
        LPAnalytics.LivePerson.trackCheckActiveConversation(brandId, error)
    }

    /**
     * Send analytics data for tracking checkConversationIsMarkedAsUrgent api usage. If SDK initialized successfully, error will be empty
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackCheckConversationIsMarkedAsUrgent(brandId: String?, error: String?) {
        LPAnalytics.LivePerson.trackCheckConversationIsMarkedAsUrgent(brandId, error)
    }

    /**
     * Send analytics data for tracking checkAgentID api usage. If SDK initialized successfully, error will be empty
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackCheckAgentID(brandId: String?, error: String?) {
        LPAnalytics.LivePerson.trackCheckAgentID(brandId, error)
    }

    /**
     * Send analytics data for tracking markConversationAsUrgent & markConversationAsNormal api usage. If SDK initialized successfully, error will be empty
     * @param isUrgent to check tracking is for markConversationAsUrgent or markConversationAsNormal
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackMarkConversation(isUrgent: Boolean,  brandId: String?, error: String?) {
        LPAnalytics.LivePerson.trackMarkConversation(isUrgent,  brandId, error)
    }

    /**
     * Send analytics data for tracking resolveConversation api usage. If SDK initialized successfully, error will be empty
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackResolveConversation(brandId: String?, error: String?) {
        LPAnalytics.LivePerson.trackResolveConversation(brandId, error)
    }

    /**
     * Send analytics data for tracking clearHistory api usage. If SDK initialized successfully, error will be empty
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackClearHistory(brandId: String?, error: String?) {
        LPAnalytics.LivePerson.trackClearHistory(brandId, error)
    }

    /**
     * Send analytics data for tracking shutdown api usage. If SDK initialized successfully, error will be empty
     * @param version method version
     * @param startUpTime Time when initialization started
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackShutdown(version: LPAPIVersion, startUpTime: Long, brandId: String?, error: String?) {
        LPAnalytics.LivePerson.trackShutdown(version, startUpTime, brandId, error)
    }

    /**
     * Send analytics data for tracking setImageServicePendingIntent api usage. If PendingIntent is null send error else error will be empty
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackSetImageServicePendingIntent(brandId: String?, error: String?) {
        LPAnalytics.LivePerson.trackSetImageServicePendingIntent(brandId, error)
    }

    /**
     * Send analytics data for tracking setImageServiceUploadNotificationBuilder api usage. If Notification Builder is null send error else error will be empty
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackSetImageServiceUploadNotificationBuilder(brandId: String?, error: String?) {
        LPAnalytics.LivePerson.trackSetImageServiceUploadNotificationBuilder(brandId, error)
    }

    /**
     * Send analytics data for tracking setImageServiceDownloadNotificationBuilder api usage. If Notification Builder is null send error else error will be empty
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackSetImageServiceDownloadNotificationBuilder(brandId: String?, error: String?) {
        LPAnalytics.LivePerson.trackSetImageServiceDownloadNotificationBuilder(brandId, error)
    }

    /**
     * Send analytics data for tracking setPushNotificationTapped api usage. If SDK initialized successfully, error will be empty
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */
    open fun trackSetPushNotificationTapped(brandId: String?, error: String?) {
        LPAnalytics.LivePerson.trackSetPushNotificationTapped(brandId, error)
    }

    /**
     * Send analytics data for tracking setLogLevel api usage.
     * @param brandId brand's account id
     * @param logLevel SDK Log level provided
     * @param isLogDump To check analytic call is for log dump of log level
     */


    open fun trackSetLogLevel(brandId: String?,logLevel: LogLevel?, isLogDump: Boolean) {
        LPAnalytics.LivePerson.trackSetLogLevel(brandId,logLevel, isLogDump)
    }

    /**
     * Send analytics data for tracking isPusherRegistered api usage. If SDK initialized successfully, error will be empty
     * @param authType Authentication parameters provided
     * @param hasDeviceToken boolean value to check if device token is provided
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */

    open fun trackIsPusherRegistered(authType: LPAuthenticationType?, hasDeviceToken:Boolean,brandId: String?, error: String?) {
        LPAnalytics.LivePerson.trackIsPusherRegistered(authType,hasDeviceToken, brandId, error)

    }

    /**
     * Send analytics data for tracking unRegisterPusher api usage. If SDK initialized successfully, error will be empty
     * @param version method version
     * @param brandId brand's account id
     * @param error Error If SDK is not initialized, null otherwise
     */

    open fun trackUnRegisterPusher(version: LPAPIVersion, brandId: String?,hasAppId: Boolean, error: String?) {
        LPAnalytics.LivePerson.trackUnRegisterPusher(version, brandId,hasAppId, error)

    }

    /**
     * Send analytics data for tracking isDebuggable api usage.
     * @param brandId brand's account id
     * @param isDebuggable boolean value to check if debuggable
     */


    open fun trackIsDebuggable(brandId: String?, isDebuggable: Boolean) {
        LPAnalytics.LivePerson.trackIsDebuggable(brandId, isDebuggable)
    }
    /**
     * Send analytics data for setDataMasking method of Logging class available in LivePerson class
     * @param brandId Host provided brandId
     * @param dataMasking boolean value to enable/disable data masking
     */
    open fun trackSetDataMasking(brandId: String?, dataMasking : Boolean) {
        LPAnalytics.LivePerson.trackSetDataMasking(brandId, dataMasking)
    }
    /**
     * Send analytics data for getLogSnapshot method of Logging class available in LivePerson class
     * @param brandId Host provided brandId
     * @param logLevel current logLevel set with in the SDK
     */
    open fun trackGetLogSnapshot(brandId: String?, logLevel: LogLevel?) {
        LPAnalytics.LivePerson.trackGetLogSnapshot(brandId,logLevel)
    }
    /**
     * Send analytics data for getLogSnapshotString method of Logging class available in LivePerson class
     * @param brandId Host provided brandId
     * @param logLevel current logLevel set with in the SDK
     */
    open fun trackGetLogSnapshotString(brandId: String?, logLevel: LogLevel?) {
        LPAnalytics.LivePerson.trackGetLogSnapshotString(brandId,logLevel)
    }

    /**
     * Send analytics data for getLogSnapshotBlock method of Logging class available in LivePerson class
     * @param brandId Host provided brandId
     * @param logLevel current logLevel set with in the SDK
     */
    open fun trackGetLogSnapshotBlock(brandId: String?, logLevel: LogLevel?) {
        LPAnalytics.LivePerson.trackGetLogSnapshotBlock(brandId,logLevel)
    }
    /**
     * Send analytics data for clearHistory method of Logging class available in LivePerson class
     * @param brandId Host provided brandId
     */
    open fun trackLoggingClearHistory(brandId: String?) {
        LPAnalytics.LivePerson.trackLoggingClearHistory(brandId)
    }

    /**
     * Send analytics data of SDK version.
     * @param brandId Host provided brandId
     * @param sdkVersion LP SDK Version in use
     */
    open fun trackGetSDKVersion(brandId: String?, sdkVersion : String?){
        LPAnalytics.LivePerson.trackGetSDKVersion(brandId,sdkVersion)
    }
}