package com.liveperson.infra.log

import java.util.concurrent.LinkedBlockingQueue

/**
 * Thread-Safe First-In First-Out Evicting Queue, specifically for use in handling logs.
 *
 * Unfortunately this class doesn't exist in Java standard libs, and we try to avoid
 * including libraries that might have such a collection, so this demi-collection will
 * have to suffice.
 */
class LoggingQueue(maxSize: Int) {

	private val backingQueue = LinkedBlockingQueue<LogLine>(maxSize)

	val size
		get() = backingQueue.size

	/**
	 * Checks if this queue is empty.
	 *
	 * @return **True** if the queue's size is 0, **False** otherwise.
	 */
	fun isEmpty(): Boolean {
		return backingQueue.isEmpty()
	}

	/**
	 * Adds the provided log line to the queue.
	 * If the queue's max capacity has been reached, evicts log lines until there is
	 * sufficient room for the new log line.
	 *
	 * @param line The new LogLine to add to the tail-end of the logs.
	 *
	 * @return **True** once the log line has been added.
	 */
	fun add(line: LogLine): Boolean {
		while (!backingQueue.offer(line)) {
			backingQueue.remove()
		}
		return true
	}

	/**
	 * Removes all log lines from this queue.
	 */
	fun clear() {
		backingQueue.clear()
	}

	/**
	 * Gets a snapshot of log history, as a List of LogLine objects, filtered by
	 * LogLevel.
	 *
	 * @param filter The highest LogLevel of lines you want included in the snapshot.
	 *
	 * @return a List<LogLine> containing all log lines at an equal or lesser logging
	 * level than `filter`, and none of the lines at a greater logging level.
	 */
	fun getLogSnapshot(filter: LogLevel): List<LogLine> {
		return backingQueue.filter { line -> line.level <= filter }
	}

	/**
	 * Gets a snapshot of log history, as a List of Strings, filtered by LogLevel.
	 *
	 * @param filter The highest LogLevel of lines you want included in the snapshot.
	 *
	 * @return a List<String> containing all log lines at an equal or lesser logging
	 * level than `filter`, and none of the lines at a greater logging level.
	 */
	fun getLogSnapshotStrings(filter: LogLevel): List<String> {
		return getLogSnapshot(filter).map { line -> line.toString() }
	}

	/**
	 * Gets a snapshot of log history, as a single String with each line separated by
	 * newline characters, filtered by LogLevel.
	 *
	 * @param filter The highest LogLevel of lines you want included in the snapshot.
	 *
	 * @return a large String containing all log lines at an equal or lesser logging
	 * level than `filter`, and none of the lines at a greater logging level.
	 */
	fun getLogSnapshotStringBlock(filter: LogLevel): String {
		val builder = StringBuilder()
		var first = true
		for (lineString in getLogSnapshotStrings(filter)) {
			if (first) {
				first = false
			} else {
				builder.append("\n")
			}
			builder.append(lineString)
		}
		return builder.toString()
	}
}
