package com.liveperson.infra;

import android.os.Parcel;
import android.os.Parcelable;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * A Parcelable POGO class that holds campaign and engagement IDs
 * Created by nirni on 12/20/17.
 */

public class CampaignInfo implements Parcelable{

	private Long mCampaignId;
	private Long mEngagementId;
	private String mSessionId;
	private String mVisitorId;
	private String mContextId;
	@Nullable
	private String mConnectorId;
	private boolean isOutboundCampaign;

	/**
	 * Construct a CampaignInfo object.
	 * campaignId, engagementId and engagementContextId are mandatory. If one of them is null or empty a {@link BadArgumentException} is thrown
	 * @param campaignId campaign ID (must be provided)
	 * @param engagementId engagement ID (must be provided)
	 * @param engagementContextId interaction context ID (must be provided)
	 * @param sessionId session ID
	 * @param visitorId visitor ID
	 * @throws BadArgumentException
	 */
	public CampaignInfo(@NonNull Long campaignId, @NonNull Long engagementId, @NonNull String engagementContextId, String sessionId, String visitorId) throws BadArgumentException {
		this(campaignId, engagementId, engagementContextId, null, sessionId, visitorId);
	}

	/**
	 * Construct a CampaignInfo object.
	 * campaignId, engagementId and engagementContextId are mandatory. If one of them is null or empty a {@link BadArgumentException} is thrown
	 * @param campaignId campaign ID (must be provided)
	 * @param engagementId engagement ID (must be provided)
	 * @param engagementContextId interaction context ID (must be provided)
	 * @param connectorId Engagement's optional connector ID
	 * @param sessionId session ID
	 * @param visitorId visitor ID
	 * @throws BadArgumentException
	 */
	public CampaignInfo(@NonNull Long campaignId, @NonNull Long engagementId, @NonNull String engagementContextId, @Nullable String connectorId, String sessionId, String visitorId) throws BadArgumentException {

		// If campaignId and engagementId provided, engagementContextId is mandatory
		//noinspection ConstantConditions Keep the null check even thought they have NonNull annotation in case customer pass null by mistake.
		if (campaignId == null|| engagementId == null ) {
			throw new BadArgumentException("campaignId and engagement cannot be null");
		}

		mCampaignId = campaignId;
		mEngagementId = engagementId;
		mContextId = engagementContextId;
		mConnectorId = connectorId;
		mSessionId = sessionId;
		mVisitorId = visitorId;
	}

	public CampaignInfo(Long campaignId, Long engagementId, boolean outboundCampaign) {
		mCampaignId = campaignId;
		mEngagementId = engagementId;
		isOutboundCampaign = outboundCampaign;
	}


	protected CampaignInfo(Parcel in) {
		if (in.readByte() == 0) {
			mCampaignId = null;
		} else {
			mCampaignId = in.readLong();
		}
		if (in.readByte() == 0) {
			mEngagementId = null;
		} else {
			mEngagementId = in.readLong();
		}
		mSessionId = in.readString();
		mVisitorId = in.readString();
		mContextId = in.readString();
		mConnectorId = in.readString();
		isOutboundCampaign = in.readInt() == 1;
	}

	public static final Creator<CampaignInfo> CREATOR = new Creator<CampaignInfo>() {
		@Override
		public CampaignInfo createFromParcel(Parcel in) {
			return new CampaignInfo(in);
		}

		@Override
		public CampaignInfo[] newArray(int size) {
			return new CampaignInfo[size];
		}
	};

	public Long getCampaignId() {
		return mCampaignId;
	}

	public void setCampaignId(Long campaignId) {
		mCampaignId = campaignId;
	}

	public Long getEngagementId() {
		return mEngagementId;
	}

	public void setEngagementId(Long engagementId) {
		mEngagementId = engagementId;
	}

	public String getSessionId() {
		return mSessionId;
	}

	public void setSessionId(String sessionId) {
		mSessionId = sessionId;
	}

	public String getVisitorId() {
		return mVisitorId;
	}

	public void setVisitorId(String visitorId) {
		mVisitorId = visitorId;
	}

	public String getContextId() {
		return mContextId;
	}

	@Nullable
	public String getConnectorId() {
		return mConnectorId;
	}

	public void setContextId(String contextId) {
		mContextId = contextId;
	}

	public void setConnectorId(@Nullable String connectorId) {
		this.mConnectorId = connectorId;
	}

	public boolean isOutboundCampaign() {
		return isOutboundCampaign;
	}


	@Override
	public int describeContents() {
		return 0;
	}

	@Override
	public void writeToParcel(Parcel dest, int flags) {

		if (mCampaignId == null) {
			dest.writeByte((byte) 0);
		} else {
			dest.writeByte((byte) 1);
			dest.writeLong(mCampaignId);
		}
		if (mEngagementId == null) {
			dest.writeByte((byte) 0);
		} else {
			dest.writeByte((byte) 1);
			dest.writeLong(mEngagementId);
		}
		dest.writeString(mSessionId);
		dest.writeString(mVisitorId);
		dest.writeString(mContextId);
		dest.writeString(mConnectorId);
		dest.writeInt(isOutboundCampaign ? 1 : 0);
	}

	@Override
	public String toString() {
		return "CampaignInfo{" +
				"mCampaignId=" + mCampaignId +
				", mEngagementId=" + mEngagementId +
				", mSessionId=" + mSessionId +
				", mVisitorId=" + mVisitorId +
				", mContextId=" + mContextId +
				", mConnectorId=" + mConnectorId +
				'}';
	}
}
