package com.liveperson.infra;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;

import androidx.annotation.VisibleForTesting;

import com.liveperson.infra.analytics.AnalyticsService;
import com.liveperson.infra.callbacks.InitLivePersonCallBack;
import com.liveperson.infra.callbacks.LogoutLivePersonCallBack;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.configuration.LptagEnvironment;
import com.liveperson.infra.controller.DBEncryptionService;
import com.liveperson.infra.database.DatabaseManager;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.loggos.Loggos;
import com.liveperson.infra.managers.ConsumerManager;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.infra.preferences.PushMessagePreferences;
import com.liveperson.infra.sdkstatemachine.InfraStateMachine;
import com.liveperson.infra.sdkstatemachine.init.InfraInitData;
import com.liveperson.infra.sdkstatemachine.logout.PreLogoutCompletionListener;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownCompletionListener;
import com.liveperson.infra.statemachine.InitProcess;
import com.liveperson.infra.statemachine.LogoutProcess;
import com.liveperson.infra.statemachine.ShutDownProcess;
import com.liveperson.infra.statemachine.StateMachineExecutor;
import com.liveperson.infra.utils.LocaleUtils;
import com.liveperson.infra.utils.ThreadPoolExecutor;

import static com.liveperson.infra.errors.ErrorCode.ERR_0000004F;

/**
 * Created by Ilya Gazman on 11/3/2015.
 */
public enum Infra implements Clearable {
    instance;

    private static final String TAG = "Infra";
    public static final String KEY_BRAND_ID = "brand_id";
    public static final String KEY_AUTH_KEY = "auth_key";
    public static final String KEY_READ_ONLY = "read_only";
    public static final String KEY_VIEW_PARAMS = "view_params";
    public static final String SDK_VERSION = "SDK_VERSION";
    private static final String FILE_PROVIDER_AUTHORITY_PREFIX = "com.liveperson.infra.provider."; // private since we don't want direct access to it

    private InternetConnectionService mConnectionService;
    private Context mAppContext;
    private Handler mAppHandler;
    private DBEncryptionService dbEncryptionService;
    private String mHostVersion;
    private Loggos mLoggos;
    private AnalyticsService analyticsService;
    private ConsumerManager consumerManager;

	/**
	 * LPTag environment that holds the relevant LPTag domain.
	 *
	 * Since LPTag uses different domain for each environment (GA, Alpha, QA), we need a way to set
	 * the environment that we work with from test host app.
	 * The default setting in this is GA (Production), but if a test app need to set it for different
	 * environment it can use the setEnvironment() method to set a different environment.
 	 */
	private LptagEnvironment mLptagEnvironment;

    InfraStateMachine stateMachine = null;

    Infra() {
        mAppHandler = new Handler(Looper.getMainLooper());
        initStateMachine();
    }

    /**
     * Will run the init process when system is starting
     */
    public void init(final Context context, final InfraInitData initData, final InitProcess entryPoint) {
        final InitProcess infraInit = new InfraInitProcess(context, initData, entryPoint);

        boolean interceptorsEnabled = context.getResources().getBoolean(R.bool.lp_interceptors_enabled);
		LPLog.INSTANCE.d(TAG, "init: Interceptors enabled: " + interceptorsEnabled);
		// Set the interceptors only if it is enabled and interceptors provided
		if (interceptorsEnabled && (initData.getInterceptors() != null)) {
			HttpHandler.addInterceptors(initData.getInterceptors().getHttpInterceptorList());
		}

		stateMachine.initSDK(infraInit);
    }

    private void initStateMachine() {
        if (stateMachine == null) {
            stateMachine = new InfraStateMachine();
            stateMachine.setStateMachineExecutor(new StateMachineExecutor(stateMachine.getTag(), stateMachine));
        }
    }


    /**
     * init the member mHostVersion and save it in Shared Preferences.
     * @param sdkVersion - if null, mHostVersion will be initiate from the saved value in Shared Pref.
     * Throw IllegalStateException in case sdkVersion is null and there is no value in Shared Pref.
     */
    private void initSDKVersion(String sdkVersion) {
        if (TextUtils.isEmpty(sdkVersion)){
            //read from sharedPref
            mHostVersion = PreferenceManager.getInstance().getStringValue(SDK_VERSION, "" , "");
            if (TextUtils.isEmpty(mHostVersion)){
                throw new IllegalStateException("must have a value for sdk-version!");
            }
        }else{
            mHostVersion = sdkVersion;
            //save in sharedPref
            PreferenceManager.getInstance().setStringValue(SDK_VERSION, "", mHostVersion);
        }
    }


    public boolean isInitialized() {
        return stateMachine != null && stateMachine.isSDKInitialized();
    }

    private void initInfra(Context context, InfraInitData initData) {
        setContext(context);
        initSDKVersion(initData != null ? initData.getSdkVersion() : null);


        if (mLoggos == null) {
            mLoggos = new Loggos();
        }
		mLptagEnvironment = new LptagEnvironment();
        if (dbEncryptionService == null) {
            dbEncryptionService = new DBEncryptionService();
        }
        mConnectionService = new InternetConnectionService();
        if (initData != null) {
            consumerManager = new ConsumerManager(context, initData.getBrandId());
        }
        PushMessagePreferences.INSTANCE.init(context);
    }

    /**
     * Initialize analytics service to log all user event during user's interaction with SDK.
     * @param applicationContext Host application context
     * @param brandId brand's account id
     * @param appId brand's app id
     */
    public void initAnalyticsService(Context applicationContext, String brandId, String appId) {
        getAnalyticsService().init(applicationContext, brandId, appId);
    }


    /**
     * Shutting down the sdk. will shut down infra and the given ShutDownProcess.
     * @param shutDownProcess - the upper process to be shut down.
     */
    public void shutDown(final ShutDownProcess shutDownProcess) {
        final ShutDownProcess shutDownInfra = new InfraShutDownProcess(shutDownProcess);
        stateMachine.shutDownSDK(shutDownInfra);
    }

    private void infraShutDown() {
        ForegroundService.getInstance().shutDown();
        SocketManager.getInstance().shutDown();
        ThreadPoolExecutor.killAll();
        DatabaseManager.getInstance().shutDown();
        mConnectionService.shutDown();
        mAppHandler.removeCallbacksAndMessages(null);
        mHostVersion = null;
    }

    /**
     * @return the context of the application
     */
    public Context getApplicationContext() {
        return mAppContext;
    }

    public void postOnMainThread(Runnable runnable) {
        if (mAppHandler != null) {
            mAppHandler.post(runnable);
        }
    }

    public Handler getApplicationHandler() {
        return mAppHandler;
    }

    /**
     * Should be used only for testing. Example: GetUnreadMessagesCountCommandTest.class
     */
    @VisibleForTesting
    public void setApplicationHandler(Handler handler) {
        mAppHandler = handler;
    }


    public String getHostVersion() {
        return mHostVersion;
    }


    public void unregisterToNetworkChanges() {
        if (mConnectionService != null){
            mConnectionService.unRegisteredReceiver();
        }
    }

    public void registerToNetworkChanges() {
        if (mConnectionService != null){
            mConnectionService.registeredReceiver();
        }
    }


    /**
     * deleting all the user data from DB and SharedPreferences.
     */
    @Override
    public void clear() {
        if (mAppContext != null) {
            PreferenceManager.getInstance().clearAll();
            PushMessagePreferences.INSTANCE.clearAll();
            Configuration.clearAll();
            getConsumerManager().clearDataAndShutdown();
            DatabaseManager.getInstance().clear();
            dbEncryptionService.clear();
            dbEncryptionService = null;
        }
    }

    public DBEncryptionService getDBEncryptionService() {
        if (dbEncryptionService == null) {
            dbEncryptionService = new DBEncryptionService();
        }
        return dbEncryptionService;
    }

    public ConsumerManager getConsumerManager() {
        return consumerManager;
    }

    /**
     * Create an instance of AnalyticsService class if not already done so and return.
     * @return analytics service instance
     */
    public AnalyticsService getAnalyticsService() {
        if (analyticsService == null) {
            analyticsService = new AnalyticsService();
        }
        return analyticsService;
    }

    /**
     * Initialize loggos If not already done so and return an instance
     * @return loggos instance
     */
    public Loggos getLoggos() {
        if (mLoggos == null) {
            mLoggos = new Loggos();
        }
        return mLoggos;
    }

	public LptagEnvironment getLptagEnvironment() {
		return mLptagEnvironment;
	}

	public void liteLogout() {
        DatabaseManager.getInstance().deleteTables();
    }

	/**
     * Logging out infra, deleting user data
     * @param context - app Context.
     * @param initData - initData for logout.
     * @param logoutProcess - upper level logout process.
     */
    public void logout(final Context context, InfraInitData initData, final LogoutProcess logoutProcess) {
        LogoutProcess infraLogoutProcess = new InfraLogoutProcess(context, initData, logoutProcess);
        stateMachine.logoutSDK(infraLogoutProcess);

    }

	public static String getFileProviderAuthorityPrefix(){
		return FILE_PROVIDER_AUTHORITY_PREFIX;
	}

    public void setContext(Context context) {
        if (context != null) {
            mAppContext = context.getApplicationContext();
        } else {
            LPLog.INSTANCE.e(TAG, ERR_0000004F, "setContext: The context cannot be null!");
        }
    }

    /**
     * Init Infra with sdk version and than initiating the upper level.
     */
    private class InfraInitProcess extends InitProcess {

        private final Context context;
        private final InfraInitData initData;
        private final InitProcess entryPoint;

        InfraInitProcess(Context context, InfraInitData initData, InitProcess entryPoint) {
            this.context = context;
            this.initData = initData;
            this.entryPoint = entryPoint;
        }

        @Override
        public void init() {
            LPLog.INSTANCE.d(TAG, "Initializing!");
            initInfra(context, initData);
            entryPoint.init();
        }

        @Override
        public InitLivePersonCallBack getInitCallback() {
            return entryPoint.getInitCallback();
        }
    }

    /**
     * Shutting down the sdk. will shut down the given upper level ShutDownProcess and than shut down infra.
     */
    private class InfraShutDownProcess extends ShutDownProcess {

        //upper level shut down process.
        private final ShutDownProcess shutDownProcess;

        InfraShutDownProcess(ShutDownProcess shutDownProcess) {
            this.shutDownProcess = shutDownProcess;
        }

        @Override
        public void shutDown(final ShutDownCompletionListener listener) {
            //first, shut down the upper level process.
            //when done, can run infra shutdown.
            shutDownProcess.shutDown(new ShutDownCompletionListener() {
                @Override
                public void shutDownCompleted() {
                    LPLog.INSTANCE.d(TAG, "Shutting down...");
                    infraShutDown();
                    listener.shutDownCompleted();
                }

				@Override
				public void shutDownFailed() {
					listener.shutDownFailed();
				}
			});
        }
    }

    /**
     * Logging out infra, deleting user data
     */
    private class InfraLogoutProcess extends LogoutProcess {

        private final Context context;
        private final LogoutProcess logoutProcess;
        private final InfraInitData initData;

        InfraLogoutProcess(Context context, InfraInitData initData, LogoutProcess logoutProcess) {
            this.context = context;
            this.logoutProcess = logoutProcess;
            this.initData = initData;
        }

        /**
         * init sdk for preLogout() processes.
         */
        @Override
        public void initForLogout() {
            initInfra(context, initData);
            logoutProcess.initForLogout();
        }

        /**
         * Things need to be done before shut down & logout.
         */
        @Override
        public void preLogout(final PreLogoutCompletionListener listener) {
            logoutProcess.preLogout(new PreLogoutCompletionListener() {
                @Override
                public void preLogoutCompleted() {
                    //no action needed in INFRA
                    listener.preLogoutCompleted();
                }

                @Override
                public void preLogoutFailed(Exception e) {
                    listener.preLogoutFailed(e);
                }
            });
        }

        /**
         * shut down Infra before logout.
         */
        @Override
        public void shutDownForLogout(final ShutDownCompletionListener listener) {
            logoutProcess.shutDownForLogout(new ShutDownCompletionListener() {
                @Override
                public void shutDownCompleted() {
                    LPLog.INSTANCE.d(TAG, "Shutting down for logout...");
                    infraShutDown();
                    listener.shutDownCompleted();
                }

				@Override
				public void shutDownFailed() {
					listener.shutDownFailed();
				}
			});
        }

        /**
         * logging out Infra after logging out from the upper level.
         * removing DB and shared preferences.
         */
        @Override
        public void logout() {
            logoutProcess.logout();
            clear();
            LocaleUtils.getInstance().clearEngagementLanguageCode();
        }

        @Override
        public LogoutLivePersonCallBack getLogoutCallback() {
            return logoutProcess.getLogoutCallback();
        }
    }
}
