package com.liveperson.infra;

import android.os.Parcel;
import android.os.Parcelable;

import com.liveperson.infra.model.LPWelcomeMessage;

/**
 * Holds parameters related to the conversation:
 * viewOnlyMode - whether to open the conversation screen with or without the enter message EditText
 * mCampaignInfo - campaign and engagement info for start conversation
 * Created by shiranr on 7/9/17.
 */

public class ConversationViewParams implements Parcelable {
    boolean viewOnlyMode = false;
	CampaignInfo mCampaignInfo;
	LPConversationsHistoryStateToDisplay mHistoryConversationsStateToDisplay = LPConversationsHistoryStateToDisplay.ALL;
	LPConversationHistoryMaxDaysDateType mHistoryConversationMaxDaysType = LPConversationHistoryMaxDaysDateType.startConversationDate;
	int mHistoryConversationsMaxDays = -1; //no limit
	private LPWelcomeMessage lpWelcomeMessage;

	public ConversationViewParams() {
		lpWelcomeMessage = new LPWelcomeMessage(null);
	}
    public ConversationViewParams(boolean readOnlyMode) {
		this();
		setReadOnlyMode(readOnlyMode);
	}

	public ConversationViewParams setReadOnlyMode(boolean readOnlyMode){
		viewOnlyMode = readOnlyMode;
		//If view only set to false but brand asked to filter only closed conversation, we ignore and set view only mode to true.
		if ( !viewOnlyMode && mHistoryConversationsStateToDisplay == LPConversationsHistoryStateToDisplay.CLOSE){
			viewOnlyMode = true;
		}
		return this;
	}

	public ConversationViewParams setCampaignInfo(CampaignInfo campaignInfo) {
		this.mCampaignInfo = campaignInfo;
		return this;
	}

	public ConversationViewParams setHistoryConversationsStateToDisplay(LPConversationsHistoryStateToDisplay historyConversationsStateToDisplay) {
		this.mHistoryConversationsStateToDisplay = historyConversationsStateToDisplay;
		if (mHistoryConversationsStateToDisplay == LPConversationsHistoryStateToDisplay.CLOSE){
			// if you see only close conversation we won't let you send messages. therefore view only mode is true.
			viewOnlyMode = true;
		}
		return this;
	}

	public ConversationViewParams setHistoryConversationMaxDaysType(LPConversationHistoryMaxDaysDateType historyConversationMaxDaysType) {
		this.mHistoryConversationMaxDaysType = historyConversationMaxDaysType;
		return this;
	}

	/**
	 * @param historyConversationsMaxDays must be larger than -1 or -1 to reset it.
	 * @return
	 */
	public ConversationViewParams setHistoryConversationsMaxDays(int historyConversationsMaxDays) {
		if (historyConversationsMaxDays >= -1){
			this.mHistoryConversationsMaxDays = historyConversationsMaxDays;
		}
		return this;
	}

	public LPConversationsHistoryStateToDisplay getHistoryConversationsStateToDisplay() {
		return mHistoryConversationsStateToDisplay;
	}

	public LPConversationHistoryMaxDaysDateType getHistoryConversationMaxDaysType() {
		return mHistoryConversationMaxDaysType;
	}

	public int getHistoryConversationsMaxDays() {
		return mHistoryConversationsMaxDays;
	}

	public boolean isViewOnlyMode() {
        return viewOnlyMode;
    }

	public CampaignInfo getCampaignInfo() {
		return mCampaignInfo;
	}

	public void clearCampaignInfo(){
		mCampaignInfo = null;
	}

	public boolean isFilterOn() {
		return mHistoryConversationsStateToDisplay != LPConversationsHistoryStateToDisplay.ALL
				||
				mHistoryConversationsMaxDays != -1;
	}

	public void setLpWelcomeMessage(LPWelcomeMessage lpWelcomeMessage) {
		this.lpWelcomeMessage = lpWelcomeMessage;
	}

	public LPWelcomeMessage getLpWelcomeMessage() {
		return lpWelcomeMessage;
	}

	@Override
	public String toString() {
		return "State To Display = "+ mHistoryConversationsStateToDisplay.name() +
				", Max Days = " + mHistoryConversationsMaxDays +
				", Max Days Type = "  + mHistoryConversationMaxDaysType.name();
	}


	//************ Parcelable ****************//

	@Override
	public int describeContents() {
		return 0;
	}

	@Override
	public void writeToParcel(Parcel dest, int flags) {
		dest.writeByte((byte) (viewOnlyMode ? 1 : 0));
		dest.writeParcelable(mCampaignInfo, flags);
		dest.writeInt(mHistoryConversationsMaxDays);
		dest.writeInt(mHistoryConversationMaxDaysType.ordinal());
		dest.writeInt(mHistoryConversationsStateToDisplay.ordinal());
		dest.writeParcelable(lpWelcomeMessage, flags);
	}

	protected ConversationViewParams(Parcel in) {
		viewOnlyMode = in.readByte() != 0;
		mCampaignInfo = in.readParcelable(CampaignInfo.class.getClassLoader());
		mHistoryConversationsMaxDays = in.readInt();
		mHistoryConversationMaxDaysType = LPConversationHistoryMaxDaysDateType.values()[in.readInt()];
		mHistoryConversationsStateToDisplay = LPConversationsHistoryStateToDisplay.values()[in.readInt()];
		lpWelcomeMessage = in.readParcelable(ConversationViewParams.class.getClassLoader());
	}

	public static final Creator<ConversationViewParams> CREATOR = new Creator<ConversationViewParams>() {
		@Override
		public ConversationViewParams createFromParcel(Parcel in) {
			return new ConversationViewParams(in);
		}

		@Override
		public ConversationViewParams[] newArray(int size) {
			return new ConversationViewParams[size];
		}
	};

}