package com.liveperson.infra;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by ofira on 16/11/2015.
 */
public class LocalBroadcastReceiver {

    private IOnReceive mReceiver;
    private LPBroadcastReceiver mLocalBroadcastManager;
    private List<String> mIntentFilters;
    private boolean isRegistered;

    /**
     * Unregister a previously registered BroadcastReceiver.
     * All filters that have been registered for this BroadcastReceiver will be
     * removed.
     */
    public void unregister() {
        isRegistered = false;
        mLocalBroadcastManager.unregister();
    }

    /**
     * Register a receive for the local broadcasts
     */
    public void register() {
        isRegistered = true;
        mLocalBroadcastManager.register();
    }

    public interface IOnReceive {
        void onBroadcastReceived(Context context, Intent intent);
    }

    private LocalBroadcastReceiver(Builder builder, IOnReceive receiverCallback) {
        mIntentFilters = builder.intentFilters;
        mReceiver = receiverCallback;
        mLocalBroadcastManager = new LPBroadcastReceiver();
        register();
    }

    public boolean isRegistered() {
        return isRegistered;
    }

    public static class Builder {
        private List<String> intentFilters = new ArrayList<>();

        public Builder addAction(String action) {
            intentFilters.add(action);
            return this;
        }

        public LocalBroadcastReceiver build(IOnReceive receiverCallback) {
            return new LocalBroadcastReceiver(this, receiverCallback);
        }
    }

    public class LPBroadcastReceiver extends BroadcastReceiver {
        private LocalBroadcastManager broadcastManager;

        public LPBroadcastReceiver() {
            broadcastManager = LocalBroadcastManager.getInstance(Infra.instance.getApplicationContext());
        }

        public void register() {
            for (String filterKey : mIntentFilters) {
                broadcastManager.registerReceiver(this, new IntentFilter(filterKey));
            }
        }


        @Override
        public void onReceive(Context context, Intent intent) {
            mReceiver.onBroadcastReceived(context, intent);
        }

        public void unregister() {
            broadcastManager.unregisterReceiver(this);
        }
    }
}
