package com.liveperson.infra.analytics.events

import android.content.Context
import com.liveperson.infra.PushType
import com.liveperson.infra.PushUnregisterType
import com.liveperson.infra.analytics.*
import com.liveperson.infra.auth.LPAuthenticationType
import com.liveperson.infra.log.LogLevel

/**
 * Class to gather LivePerson facade's analytics data.
 */
open class LPAnalyticsFacade(private val analyticsService: AnalyticsService) {

    companion object {
        private const val API_VERSION = "api_version"
        private const val STARTUP_TIME = "startup_time"
        private const val EXECUTION_TIME = "execution_time"
        private const val AUTH_TYPE = "auth_type"
        private const val HAS_CAMPAIGN_INFO = "has_campaign_info"
        private const val HAS_NOTIFICATION_TOKEN = "has_notification_token"
        private const val HAS_APP_ID = "has_app_id"
        private const val SHOW_NOTIFICATION = "show_notification"
        private const val IS_OUTBOUND_CAMPAIGN_PUSH = "is_outbound_campaign_push"
        private const val BACKEND_SERVICE = "backend_service"
        private const val FORCE_LOGOUT = "force_logout"
        private const val PUSH_UNREGISTER_TYPE = "push_unregister_type"
        private const val PUSH_PLATFORM = "push_platform"
        private const val Error = "error"
        private const val DEBUGGABLE = "debuggable"
        private const val DATA_MASKING = "data_masking"
        private const val HAS_FIRST_NAME = "has_first_name"
        private const val HAS_LAST_NAME = "has_last_name"
        private const val HAS_PHONE_NUMBER = "has_phone_number"
        private const val HAS_CALLBACK = "is_callback_listener"
        private const val LOG_LEVEL = "log_level"
        private const val LP_SDK_VERSION = "lp_sdk_version"
    }

    /**
     * Gather analytics data for LivePerson facade's initialize method.
     * @param version method version to track If deprecated one is still in use
     * @param startUpTime Time it took to complete initialization process
     * @param error Error If failed to initialize. Null otherwise
     */
    fun trackInit(version: LPAPIVersion, startUpTime: Long, error: String?) {
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.INITIALIZE,
                EventProperty(API_VERSION, version.value),
                EventProperty(STARTUP_TIME, startUpTime),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's showConversation method.
     * @param version method version to track If deprecated one is still in use
     * @param authType Type of authentication being used by customers
     * @param hasCampaignInfo Is customer passing in campaign info
     * @param error Error If failed to show. Null otherwise
     */
    fun trackDisplayConversation(version: LPAPIVersion, authType: LPAuthenticationType?, hasCampaignInfo: Boolean, isUsingFragment: Boolean, error: String?) {
        val eventName = if (isUsingFragment) EventName.LivePerson.SHOW_CONVERSATION_HOST_CONTROL else EventName.LivePerson.SHOW_CONVERSATION_LP_CONTROL
        analyticsService.logUserEvent(AnalyticsEvent(eventName,
                EventProperty(API_VERSION, version.value),
                EventProperty(AUTH_TYPE, authType),
                EventProperty(HAS_CAMPAIGN_INFO, hasCampaignInfo),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's registerLpPusher method.
     * @param version method version to track If deprecated one is still in use
     * @param brandId Brand's account id
     * @param authType Type of authentication being used by customers
     * @param pushType Type of push notification.
     * @param isPushTokenProvided Is push notification/gcm token provided by host app
     * @param error Error If failed to register. Null otherwise
     */
    fun trackRegisterLPPusher(version: LPAPIVersion, brandId: String?, pushType: PushType, authType: LPAuthenticationType?, isPushTokenProvided: Boolean, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.REGISTER_PUSHER,
                EventProperty(API_VERSION, version.value),
                EventProperty(AUTH_TYPE, authType),
                EventProperty(PUSH_PLATFORM, pushType.platform),
                EventProperty(HAS_NOTIFICATION_TOKEN, isPushTokenProvided),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's getUnreadMessageCount method.
     * @param version method version to track If deprecated one is still in use
     * @param brandId Brand's account id
     * @param hasAppId If host app has provided appId
     * @param authType Type of authentication being used by customers
     * @param callBackAvailable boolean value to check whether callback was assigned to the method
     * @param error Error If failed to get count. Null otherwise
     */
    fun trackGetUnreadMessageCount(version: LPAPIVersion, brandId: String?, hasAppId: Boolean, authType: LPAuthenticationType?, callBackAvailable: Boolean, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.GET_UNREAD_MESSAGES_COUNT,
                EventProperty(API_VERSION, version.value),
                EventProperty(AUTH_TYPE, authType),
                EventProperty(HAS_CALLBACK, callBackAvailable),
                EventProperty(HAS_APP_ID, hasAppId),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's reconnect method.
     * @param version method version to track If deprecated one is still in use
     * @param brandId Brand's account id
     * @param authType Type of authentication being used by customers
     * @param error Error If failed to reconnect. Null otherwise
     */
    fun trackReconnect(version: LPAPIVersion, brandId: String?, authType: LPAuthenticationType?, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.RECONNECT,
                EventProperty(API_VERSION, version.value),
                EventProperty(AUTH_TYPE, authType),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's logout method.
     * @param version method version to track If deprecated one is still in use
     * @param brandId Brand's account id
     * @param hasAppId If host app has provided appId during logout
     * @param executionTime time taken to execute the logout operation
     * @param forceLogOut SDK won't wait for unregister pusher succeed before logout when it's true
     * @param type pusher unregister type request (AGENT/ALL/NONE)
     * @param error Error If failed to logout. Null otherwise
     */
    fun trackLogout(version: LPAPIVersion, brandId: String?, executionTime: Long, hasAppId: Boolean, forceLogOut: Boolean, type: PushUnregisterType?, error: String?) {
        analyticsService.setBrandId(brandId)
        val eventProperties = ArrayList<EventProperty>()
        eventProperties.add(EventProperty(API_VERSION, version.value))
        eventProperties.add(EventProperty(EXECUTION_TIME, executionTime))
        eventProperties.add(EventProperty(HAS_APP_ID, hasAppId))
        eventProperties.add(EventProperty(FORCE_LOGOUT, forceLogOut))
        if (!type?.name.isNullOrEmpty()) {
            eventProperties.add(EventProperty(PUSH_UNREGISTER_TYPE, type?.name))
        }
        if (!error.isNullOrEmpty()) {
            eventProperties.add(EventProperty(Error, error))
        }
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.LOGOUT, *eventProperties.toTypedArray()))
    }

    /**
     * Gather analytics data for LivePerson facade's handlePushMessage method.
     * @param version method version to track If deprecated one is still in use
     * @param brandId Brand's account id
     * @param appContext application context received from brands app to SDK
     * @param showNotification If host app has given permission to SDK to display notification
     * @param isOutboundCampaignPush If notification is originated as a part of outbound campaign
     * @param backendService Backend service where this notification has originated from
     * @param error Error If failed handle. Null otherwise
     */
    fun trackHandlePushMessage(version: LPAPIVersion, appContext: Context?, brandId: String?, showNotification: Boolean, isOutboundCampaignPush: Boolean, backendService: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.setAppContext(appContext)
        val eventProperties = ArrayList<EventProperty>()
        eventProperties.add(EventProperty(IS_OUTBOUND_CAMPAIGN_PUSH, isOutboundCampaignPush))
        eventProperties.add(EventProperty(API_VERSION, version.value))
        eventProperties.add(EventProperty(SHOW_NOTIFICATION, showNotification))
        if (!backendService.isNullOrEmpty()) {
            eventProperties.add(EventProperty(BACKEND_SERVICE, backendService))
        }
        if (!error.isNullOrEmpty()) {
            eventProperties.add(EventProperty(Error, error))
        }
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.HANDLE_PUSH, *eventProperties.toTypedArray()))
    }

    /**
     * Gather analytics data for LivePerson facade's hideConversation method.
     * @param version method version to track If deprecated one is still in use
     * @param brandId Brand's account id
     * @param authType Type of authentication being used by customers
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackHideConversation(version: LPAPIVersion, authType: LPAuthenticationType?, brandId: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.HIDE_CONVERSATION,
                EventProperty(API_VERSION, version.value),
                EventProperty(AUTH_TYPE, authType),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's updateTokenInBackground method.
     * @param brandId Brand's account id
     * @param authType Type of authentication being used by customers
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackUpdateTokenInBackground(authType: LPAuthenticationType?, brandId: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.UPDATE_TOKEN_IN_BACKGROUND,
                EventProperty(AUTH_TYPE, authType),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's setCallback & removeCallback method.
     * @param isSetCallback method version to track If setCallback or removeCallback method is called
     * @param isListenerAvailable contains if listener provided to serCallback method for removeCallback ignored
     * @param brandId Brand's account id
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackSetCallback(brandId: String?, isListenerAvailable: Boolean, error: String?, isSetCallback: Boolean) {
        analyticsService.setBrandId(brandId)
        if (isSetCallback)
            analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.SET_CALLBACK,
                    EventProperty(HAS_CALLBACK, isListenerAvailable),
                    EventProperty(Error, error)))
        else
            analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.REMOVE_CALLBACK,
                    EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's setUserProfile method.
     * @param version method version to track If deprecated one is still in use
     * @param brandId Brand's account id
     * @param hasAppId If host app has provided appId during logout
     * @param hasFirstName Customer firstName has been provided
     * @param hasLastName Customer lastName has been provided
     * @param hasPhoneNumber Customer phone number has been provided
     * @param error Error If failed handle. Null otherwise
     */

    open fun trackSetUserProfile(version: LPAPIVersion, brandId: String?, hasAppId: Boolean, hasFirstName: Boolean, hasLastName: Boolean, hasPhoneNumber: Boolean, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.SET_USER_PROFILE,
                EventProperty(API_VERSION, version.value),
                EventProperty(HAS_APP_ID, hasAppId),
                EventProperty(HAS_FIRST_NAME, hasFirstName),
                EventProperty(HAS_LAST_NAME, hasLastName),
                EventProperty(HAS_PHONE_NUMBER, hasPhoneNumber),
                EventProperty(Error, error)))
    }


    /**
     * Gather analytics data for LivePerson facade's checkActiveConversation method.
     * @param brandId Brand's account id
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackCheckActiveConversation(brandId: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.CHECK_ACTIVE_CONVERSATION,
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's checkConversationIsMarkedAsUrgent method.
     * @param brandId Brand's account id
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackCheckConversationIsMarkedAsUrgent(brandId: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.CHECK_CONVERSATION_MARKED_URGENT,
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's checkAgentId method.
     * @param brandId Brand's account id
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackCheckAgentID(brandId: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.CHECK_AGENT_ID,
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's markConversationUrgent & markConversationNormal method.
     * @param isUrgent to check whether track event is for markConversationUrgent Or markConversationNormal
     * @param brandId Brand's account id
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackMarkConversation(isUrgent: Boolean, brandId: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        if (isUrgent)
            analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.MARK_CONVERSATION_URGENT,
                    EventProperty(Error, error)))
        else
            analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.MARK_CONVERSATION_NORMAL,
                    EventProperty(Error, error)))

    }

    /**
     * Gather analytics data for LivePerson facade's resolveConversation method.
     * @param brandId Brand's account id
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackResolveConversation(brandId: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.RESOLVE_CONVERSATION,
                EventProperty(Error, error)))
    }


    /**
     * Gather analytics data for LivePerson facade's clearHistory method.
     * @param brandId Brand's account id
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackClearHistory(brandId: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.CLEAR_HISTORY,
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's shutdown method.
     * @param version method version to track If deprecated one is still in use
     * @param startUpTime Time it took to complete initialization process
     * @param brandId Brand's account id
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackShutdown(version: LPAPIVersion, startUpTime: Long, brandId: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.SHUTDOWN,
                EventProperty(API_VERSION, version.value),
                EventProperty(STARTUP_TIME, startUpTime),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's setImageServicePendingIntent method.
     * @param brandId Brand's account id
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackSetImageServicePendingIntent(brandId: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.SET_IMAGE_SERVICE_PENDING_INTENT,
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's setImageServiceUploadNotificationBuilder method.
     * @param brandId Brand's account id
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackSetImageServiceUploadNotificationBuilder(brandId: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.SET_IMAGE_SERVICE_UPLOPAD_NOTIFICATION_BUILDER,
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's setImageServiceDownloadNotification method.
     * @param brandId Brand's account id
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackSetImageServiceDownloadNotificationBuilder(brandId: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.SET_IMAGE_SERVICE_DOWNLOAD_NOTIFICATION_BUILDER,
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's setPushNotificationTapped method.
     * @param brandId Brand's account id
     * @param isOutboundCampaignPush If notification is originated as a part of outbound campaign
     * @param backendService Backend service where this notification has originated from
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackSetPushNotificationTapped(version: LPAPIVersion, brandId: String?, isOutboundCampaignPush: Boolean, backendService: String?, error: String?) {
        analyticsService.setBrandId(brandId)
        val eventProperties = ArrayList<EventProperty>()
        eventProperties.add(EventProperty(IS_OUTBOUND_CAMPAIGN_PUSH, isOutboundCampaignPush))
        eventProperties.add(EventProperty(API_VERSION, version.value))
        if (!backendService.isNullOrEmpty()) {
            eventProperties.add(EventProperty(BACKEND_SERVICE, backendService))
        }
        if (!error.isNullOrEmpty()) {
            eventProperties.add(EventProperty(Error, error))
        }
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.SET_PUSH_NOTIFICATIOM_TAPPED, *eventProperties.toTypedArray()))
    }

    /**
     * Gather analytics data for LivePerson facade's to track setLogLevel method.
     *  @param brandId Brand's account id
     * @param logLevel SDK log level in use
     * @param isLogDump to check track request is for log dump or log level
     */
    open fun trackSetLogLevel(brandId: String?, logLevel: LogLevel?, isLogDump: Boolean) {
        analyticsService.setBrandId(brandId)
        if (isLogDump) {
            analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.GET_LOG_LEVEL, EventProperty(LOG_LEVEL, logLevel?.apiName)))
        } else {
            analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.SET_LOG_LEVEL, EventProperty(LOG_LEVEL, logLevel?.apiName)))
        }
    }

    /**
     * Gather analytics data for LivePerson facade's to track isPusherRegistered Method.
     * @param authType Type of authentication being used by customers
     * @param hasDeviceToken boolean value to check if device token is available
     * @param brandId Brand's account id
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackIsPusherRegistered(authType: LPAuthenticationType?, hasDeviceToken: Boolean, brandId: String?, error: String?) {

        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.IS_PUSHER_REGISTERED,
                EventProperty(AUTH_TYPE, authType),
                EventProperty(HAS_NOTIFICATION_TOKEN, hasDeviceToken),
                EventProperty(Error, error)))


    }

    /**
     * Gather analytics data for LivePerson facade's to track UnRegisterPusher Method.
     * @param version method version to track If deprecated one is still in use
     * @param brandId Brand's account id
     * @param error Error If failed handle. Null otherwise
     */
    open fun trackUnRegisterPusher(version: LPAPIVersion, brandId: String?, hasAppId: Boolean, error: String?) {
        analyticsService.setBrandId(brandId)

        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.UNREGISTER_PUSHER,
                EventProperty(API_VERSION, version.value),
                EventProperty(HAS_APP_ID, hasAppId),
                EventProperty(Error, error)))
    }

    /**
     * Gather analytics data for LivePerson facade's to track isDebuggable Method.
     * @param brandId Brand's account id
     * @param isDebuggable boolean value to check isDebuggable
     */
    open fun trackIsDebuggable(brandId: String?, isDebuggable: Boolean) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.IS_DEBUGGABLE,
                EventProperty(DEBUGGABLE, isDebuggable)))
    }

    /**
     * Gather analytics data for LivePerson facade's to track setDataMasking Method of Logging class present in LivePerson class.
     * @param brandId Brand's account id
     * @param dataMasking SDK data masking value
     */
    open fun trackSetDataMasking(brandId: String?, dataMasking: Boolean) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.SET_DATA_MASKING,
                EventProperty(DATA_MASKING, dataMasking)))
    }

    /**
     * Gather analytics data for LivePerson facade's to track getLogSnapshot Method of Logging class present in LivePerson class.
     * @param brandId Brand's account id
     * @param logLevel SDK Logging level
     */
    open fun trackGetLogSnapshot(brandId: String?, logLevel: LogLevel?) {

        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.GET_LOG_SNAPSHOT,
                EventProperty(LOG_LEVEL, logLevel?.apiName)))
    }

    /**
     * Gather analytics data for LivePerson facade's to track getLogSnapshotString Method of Logging class present in LivePerson class.
     * @param brandId Brand's account id
     * @param logLevel SDK Logging level
     */
    open fun trackGetLogSnapshotString(brandId: String?, logLevel: LogLevel?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.GET_LOG_SNAPSHOT_STRING, EventProperty(LOG_LEVEL, logLevel?.apiName)))
    }

    /**
     * Gather analytics data for LivePerson facade's to track getLogSnapshotBlock Method of Logging class present in LivePerson class.
     * @param brandId Brand's account id
     * @param logLevel SDK Logging level
     */
    open fun trackGetLogSnapshotBlock(brandId: String?, logLevel: LogLevel?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.GET_LOG_SNAPSHOT_BLOCK, EventProperty(LOG_LEVEL, logLevel?.apiName)))
    }

    /**
     * Gather analytics data for LivePerson facade's to track clearHistory Method of Logging class present in LivePerson class.
     * @param brandId Brand's account id
     */
    open fun trackLoggingClearHistory(brandId: String?) {

        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.LOGGING_CLEAR_HISTORY))
    }

    /**
     * Gather analytics data for LivePerson facade's to track getSDKVersion Method.
     * @param brandId Brand's account id
     * @param sdkVersion String value contains LivePerson SDK Version
     */
    open fun trackGetSDKVersion(brandId: String?, sdkVersion: String?) {
        analyticsService.setBrandId(brandId)
        analyticsService.logUserEvent(AnalyticsEvent(EventName.LivePerson.GET_SDK_VERSION, EventProperty(LP_SDK_VERSION, sdkVersion)))
    }
}