package com.liveperson.infra.network.socket;

import com.liveperson.infra.IDisposable;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.socket.state.SocketStateListener;

import java.util.ArrayList;
import java.util.List;


/**
 * Created by Ilya Gazman on 11/2/2015.
 */
public class SocketStateManager implements IDisposable {

    private static final String TAG = "SocketStateManager";
    private List<SocketStateListener> mCallbacks = new ArrayList<>();
    private SocketState mState = SocketState.INIT;

    void setState(SocketState state) {
        synchronized (this) {
            if (mState == state) {
                LPLog.INSTANCE.d(TAG, "Request state and current state are equal");
                return;
            }
            LPLog.INSTANCE.d(TAG, "new socket state " + state.name());
            mState = state;
            //We have to copy this array, cause in the callback we sometimes unregister the callback
            // and mCallbacks is changing during the loop. we must not be effected from this modification.
            List<SocketStateListener> mCopyCallbacks = new ArrayList<>(mCallbacks);
            for (SocketStateListener callback : mCopyCallbacks) {
                LPLog.INSTANCE.d(TAG, "Calling listener " + callback.toString() + " new socket state " + state.name());
                callback.onStateChanged(mState);
            }
        }
    }

    void disconnected(String reason, int code){
        LPLog.INSTANCE.d(TAG, "onDisconnected, reason " + reason + " code " + code);
        synchronized (this){
            //We have to copy this array, cause in the callback we sometimes unregister the callback
            // and mCallbacks is changing during the loop. we must not be effected from this modification.
            List<SocketStateListener> mCopyCallbacks = new ArrayList<>(mCallbacks);
            for (SocketStateListener callback : mCopyCallbacks) {
                callback.onDisconnected(reason, code);
            }
        }
    }


    public void register(SocketStateListener callback) {
        LPLog.INSTANCE.i(TAG, "register SocketStateListener " + callback.hashCode());
        synchronized (this) {
            if (!mCallbacks.contains(callback)){
                mCallbacks.add(callback);
            }
        }
    }

    public void unregister(SocketStateListener callback) {
        LPLog.INSTANCE.i(TAG, "unregister SocketStateListener " + callback.hashCode());
        synchronized (this) {
            mCallbacks.remove(callback);
        }
    }

    public SocketState getState() {
        LPLog.INSTANCE.d(TAG, "getState");
        synchronized (this) {
            return mState;
        }
    }

    @Override
    public void dispose() {
        LPLog.INSTANCE.i(TAG, "dispose");
        synchronized (this) {
            mCallbacks.clear();
        }
    }
}
